<?php

namespace FirstpointCh\Shop\Payment\Gateway;

use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Payment\Enums\GatewayStatus;
use Illuminate\Http\Request;
use Log;

class StripeGateway extends PaymentGateway
{
    public string $slug = 'stripe';

    public string $name = 'Stripe';

    public string $description = 'Stripe';

    public $event;

    public function fromRequest(Request $request): self
    {
        \Stripe\Stripe::setApiKey($this->config['secret_key']);

        try {
            $this->event = \Stripe\Webhook::constructEvent(
                @file_get_contents('php://input'),
                $request->header('Stripe-Signature'),
                $this->config['webhook_secret'],
            );
        } catch (\UnexpectedValueException $e) {
            throw new \Exception('Invalid payload.');
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            throw new \Exception('Invalid signature.');
        }

        Log::info('Stripe event', [
            'event' => $this->event->data->object,
        ]);

        $this->payment = Payment::find($this->event->data->object->metadata->payment_id);

        return $this;
    }

    public function redirect(?string $cancelUrl = null)
    {
        \Stripe\Stripe::setApiKey($this->config['secret_key']);

        $paymentIntent = \Stripe\PaymentIntent::create([
            'amount' => (int) ($this->payment->amount * 100),
            'currency' => $this->payment->currency,
            'automatic_payment_methods' => [
                'enabled' => true,
            ],
            'metadata' => [
                'payment_id' => $this->payment->id,
            ],
        ]);

        return view('shop::payment.stripe.form', [
            'clientSecret' => $paymentIntent->client_secret,
            'publicKey' => $this->config['public_key'],
            'cancelUrl' => 'todo-cancel',
        ]);
    }

    public function handle(Request $request): GatewayStatus
    {
        if ($this->event->type === 'payment_intent.succeeded') {
            return GatewayStatus::Charged;
        }

        return GatewayStatus::Unknown;
    }
}
