<?php

namespace FirstpointCh\Shop\Payment\Gateway;

use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Payment\Enums\GatewayStatus;
use Illuminate\Http\Request;

class StripeCheckoutGateway extends PaymentGateway
{
    public string $slug = 'stripe-checkout';

    public string $name = 'Stripe Checkout';

    public string $description = 'Stripe';

    public $event;

    public function fromRequest(Request $request): self
    {
        \Stripe\Stripe::setApiKey($this->config['secret_key']);

        try {
            $this->event = \Stripe\Webhook::constructEvent(
                @file_get_contents('php://input'),
                $request->header('Stripe-Signature'),
                $this->config['webhook_secret'],
            );
        } catch (\UnexpectedValueException $e) {
            throw new \Exception('Invalid payload.');
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            throw new \Exception('Invalid signature.');
        }

        $this->payment = Payment::find($this->event->data->object->client_reference_id);

        return $this;
    }

    public function redirect(?string $cancelUrl = null)
    {
        \Stripe\Stripe::setApiKey($this->config['secret_key']);

        $session = \Stripe\Checkout\Session::create([
            'line_items' => [[
                'price_data' => [
                    'currency' => $this->payment->currency,
                    'unit_amount' => (int) ($this->payment->amount * 100),
                    'product_data' => [
                        'name' => 'Panier #'.$this->payment->cart_id,
                    ],
                ],
                'quantity' => 1,
            ]],
            'customer_email' => customer()->email,
            'client_reference_id' => $this->payment->reference,
            'mode' => 'payment',
            'success_url' => route('shop::payment.confirmation', [
                'gateway' => $this->slug,
                'reference' => $this->payment->reference,
            ]),
            'cancel_url' => $cancelUrl,
        ]);

        return redirect($session->url);
    }

    public function handle(Request $request): GatewayStatus
    {
        // The event property was set in the "fromRequest" method.
        if ($this->event->type === 'checkout.session.completed' && $this->event->data->object->payment_status === 'paid') {
            return GatewayStatus::Charged;
        }

        return GatewayStatus::Unknown;
    }
}
