<?php

namespace FirstpointCh\Shop\Payment\Actions;

use FirstpointCh\Shop\Actions\Action;
use FirstpointCh\Shop\Models\Cart;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Order\Enums\OrderItemStatus;
use FirstpointCh\Shop\Order\Enums\OrderStatus;
use FirstpointCh\Shop\Payment\Enums\PaymentStatus;
use Illuminate\Support\Facades\DB;

/**
 * @method static run(Payment $payment, OrderStatus $orderStatus)
 */
class PaymentIntentSuccessAction
{
    use Action;

    public function handle(Payment $payment, OrderStatus $orderStatus)
    {
        return DB::transaction(function () use ($payment, $orderStatus) {
            $cart = $payment->cart;

            // Create order
            $order = Order::create([
                'customer_id' => $cart->customer_id,
                'first_name' => $cart->first_name,
                'last_name' => $cart->last_name,
                'email' => $cart->customer->email,
                'company' => $cart->company,
                'street' => $cart->street,
                'street_number' => $cart->street_number,
                'zip' => $cart->zip,
                'city' => $cart->city,
                'country' => $cart->country,
                'billing_company' => $cart->different_billing_address ? $cart->billing_company : $cart->company,
                'billing_street' => $cart->different_billing_address ? $cart->billing_street : $cart->street,
                'billing_street_number' => $cart->different_billing_address ? $cart->billing_street_number : $cart->street_number,
                'billing_zip' => $cart->different_billing_address ? $cart->billing_zip : $cart->zip,
                'billing_city' => $cart->different_billing_address ? $cart->billing_city : $cart->city,
                'billing_country' => $cart->different_billing_address ? $cart->billing_country : $cart->country,
                'shipping_price' => $cart->getShippingMethodPrice(),
                'payment_method_price' => $cart->getPaymentMethodPrice(),
                'amount' => $payment->amount,
                'currency' => $payment->currency,
                'status' => $orderStatus,
                'payment_method' => $cart->payment_method,
                'shipping_method' => $cart->shipping_method,
                'region_id' => $cart->region_id,
            ]);

            // Create order items
            $cart->items->each(function ($item) use ($order) {
                $order->items()->create([
                    'sku' => $item->orderable->sku,
                    'name' => $item->name,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->orderable->price,
                    'total_tax_rate' => $item->orderable->product->total_tax_rate, // TODO: Test
                    'status' => OrderItemStatus::Pending,
                ]);

                if ($item->orderable->track_stock) {
                    $item->orderable->decrement('stock', $item->quantity);
                }
            });

            $cart->taxes->each(fn ($tax) => $order->taxes()->create([
                'name' => $tax->taxRule->name,
                'tax_included' => $tax->taxRule->tax_included,
                'taxable_amount' => $tax->taxable_amount,
                'rate' => $tax->taxRule->rate,
                'tax_amount' => $tax->taxRule->calculate($tax->taxable_amount),
            ]));

            // Update cart
            $cart->update([
                'order_id' => $order->id,
            ]);

            // Update payment
            $payment->update([
                'order_id' => $order->id,
                'status' => PaymentStatus::Paid,
                'status_message' => null,
            ]);

            return $order;
        });
    }
}
