<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Cart\UpdateCartTaxesAction;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class CartItem extends Model
{
    use HasPackageFactory;

    protected $guarded = [];

    protected $appends = ['is_orderable', 'price'];

    public $incrementing = false;

    protected $keyType = 'string';

    protected $casts = [
        'price' => 'float',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->id)) {
                $model->id = Str::uuid()->toString();
            }
        });

        static::created(function ($model) {
            UpdateCartTaxesAction::run($model->cart);
        });

        static::updated(function ($model) {
            UpdateCartTaxesAction::run($model->cart);
        });

        static::deleted(function ($model) {
            UpdateCartTaxesAction::run($model->cart);
        });
    }

    public function cart(): BelongsTo
    {
        return $this->belongsTo(Cart::class);
    }

    public function orderable()
    {
        return $this->morphTo()->withPrice();
    }

    public function getIsOrderableAttribute(): bool
    {
        return $this->orderable->isOrderable($this->quantity);
    }

    public function getPriceAttribute()
    {
        return $this->orderable->price * $this->quantity;
    }
}
