<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Models\Region;
use Inertia\Inertia;

class OrderController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Order/Index', [
            'orders' => $datatable
                ->searchable(['first_name', 'last_name', 'email'])
                ->query(
                    Order::with('activities')
                        ->when(request('filters.status'), fn ($query, $status) => $query->where('status', $status))
                        ->when(request('filters.region_id'), fn ($query, $regionId) => $query->where('region_id', $regionId))
                )
                ->get(),
            'datatableState' => $datatable->getState(),
            'regions' => Region::all(),
        ]);
    }

    public function show(Order $order)
    {
        money()->setFormat($order->currency);

        $order->load(['customer', 'payments', 'items', 'taxes', 'activities', 'shipments']);

        return Inertia::render('Order/Show', [
            'order' => $order,
            'subtotal' => money()->format($order->items->subtotal()),
            'total' => money()->format($order->totalAmount()),
            'carriers' => app('shop::carriers')->all(),
        ]);
    }

    public function update(Order $order)
    {
        $data = request()->validate([
            'status' => 'required|in:pending,paid,shipped,cancelled',
        ]);

        $order->update($data);

        return redirect()->back()->with('toast', [
            'title' => 'Commande enregistrée',
            'message' => 'La commande a bien été enregistrée',
            'type' => 'success',
        ]);
    }

    public function updateBillingAddress(Order $order)
    {
        $order->update(request()->validate([
            'billing_company' => 'nullable',
            'billing_street' => 'required',
            'billing_street_number' => 'required',
            'billing_zip' => 'required',
            'billing_city' => 'required',
            'billing_country' => 'required|size:2',
        ]));

        return redirect()->back()->with('toast', [
            'title' => 'Adresse mise à jour',
            'message' => 'L\'adresse a bien été mise à jour',
            'type' => 'success',
        ]);
    }

    public function updateAddress(Order $order)
    {
        $order->update(request()->validate([
            'company' => 'nullable',
            'street' => 'required',
            'street_number' => 'required',
            'zip' => 'required',
            'city' => 'required',
            'country' => 'required|size:2',
        ]));

        return redirect()->back()->with('toast', [
            'title' => 'Adresse mise à jour',
            'message' => 'L\'adresse a bien été mise à jour',
            'type' => 'success',
        ]);
    }
}
