<?php

use FirstpointCh\Shop\Models\Region;
use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the shipping method create form', function () {
    $this->get('/cp/shipping-methods/create')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ShippingMethod/Form')
            ->has('regions', 1)
        );
});

test('the name is required and must be an array', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods', ['name' => 'test'])->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods', ['name' => ['en' => 'test']])->assertSessionDoesntHaveErrors('name');
});

test('the description is required and must be an array', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods', ['description' => 'test'])->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods', ['description' => ['en' => 'test']])->assertSessionDoesntHaveErrors('description');
});

test('the slug is required', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('slug');
});

test('regions must be an array', function () {
    $this->post('/cp/shipping-methods', ['regions' => 'test'])->assertStatus(302)->assertSessionHasErrors('regions');
    $this->post('/cp/shipping-methods', ['regions' => []])->assertStatus(302)->assertSessionDoesntHaveErrors('regions');
});

test('each region must exist and have a price', function () {
    $region = Region::factory()->create();

    $this->post('/cp/shipping-methods', [
        'regions' => [['region_id' => 123, 'price' => null]],
    ])->assertStatus(302)->assertSessionHasErrors(['regions.0.region_id', 'regions.0.price']);

    $this->post('/cp/shipping-methods', [
        'regions' => [['region_id' => $region->id, 'price' => 10]],
    ])->assertStatus(302)->assertSessionDoesntHaveErrors(['regions.0.region_id', 'regions.0.price']);
});

it('stores a new shipping method', function () {
    $data = [
        'name' => ['en' => 'My shipping method'],
        'description' => ['en' => 'My shipping method'],
        'slug' => 'my-shipping-method',
        'regions' => [
            ['region_id' => defaultRegion()->id, 'min_price' => null, 'max_price' => 50,'price' => 123],
            ['region_id' => defaultRegion()->id, 'min_price' => 51, 'max_price' => 100,'price' => 200],
        ],
    ];

    $this->assertDatabaseCount('shipping_methods', 0);
    $this->assertDatabaseCount('region_shipping_method', 0);

    $this->post('/cp/shipping-methods', $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/shipping-methods');

    $regions = Arr::pull($data, 'regions');
    $data['name'] = json_encode($data['name']);
    $data['description'] = json_encode($data['description']);

    $this->assertDatabaseCount('shipping_methods', 1);
    $this->assertDatabaseCount('region_shipping_method', 2);
    $this->assertDatabaseHas('shipping_methods', $data);
    $this->assertDatabaseHas('region_shipping_method', [
        'region_id' => $regions[0],
        'shipping_method_id' => ShippingMethod::first()->id,
        'min_price' => null,
        'max_price' => 50,
        'price' => 123,
    ]);
    $this->assertDatabaseHas('region_shipping_method', [
        'region_id' => $regions[0],
        'shipping_method_id' => ShippingMethod::first()->id,
        'min_price' => 51,
        'max_price' => 100,
        'price' => 200,
    ]);
});
