<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Cart\Orderable;
use FirstpointCh\Shop\Catalog\QueryBuilders\VariantQueryBuilder;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

/**
 * @method static \FirstpointCh\Shop\Catalog\QueryBuilders\VariantQueryBuilder query()
 */
class Variant extends Model implements Orderable
{
    use HasFactory, Translatable, SoftDeletes;

    protected $guarded = [];

    protected $casts = [
        'is_active' => 'boolean',
        'name' => Localized::class,
        'track_stock' => 'boolean',
        'is_shippable' => 'boolean',
        'slug' => Localized::class,
    ];

    protected static function booted()
    {
        static::addGlobalScope('region-aware', function ($builder) {
            $builder->when(shop()->isRegionAware(), function ($query) {
                $query->whereRelation('regions', 'region_id', shop()->getRegion()->id);
            });
        });
    }

    public function newEloquentBuilder($query)
    {
        return new VariantQueryBuilder($query);
    }

    public function resolveRouteBinding($value, $field = null)
    {
        if ($field === 'slug') {
            return $this->where('slug->'.app()->getLocale(), $value)->first()
                ?? $this->where('slug->'.config('app.fallback_locale'), $value)->firstOrFail();
        }

        return $this->resolveRouteBindingQuery($this, $value, $field)->first();
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    public function attributes()
    {
        $locale = app()->getLocale();

        if ($this->connection === 'sqlite') {
            $optionValueQuery = "json_extract(value, '$.\"{$locale}\"')";
            $translatedValueQuery = "json_extract(attribute_variant.value_translated, '$.\"{$locale}\"')";
        } else {
            $optionValueQuery = "json_unquote(json_extract(value, '$.\"{$locale}\"'))";
            $translatedValueQuery = "json_unquote(json_extract(attribute_variant.value_translated, '$.\"{$locale}\"'))";
        }

        return $this->belongsToMany(Attribute::class)
            ->addSelect(DB::raw(<<<SQL
                attributes.*,
                CASE attributes.type
                    WHEN 'options' THEN (SELECT {$optionValueQuery} FROM attribute_options WHERE id = attribute_variant.value_option_id)
                    WHEN 'translated' THEN {$translatedValueQuery}
                    WHEN 'text' THEN attribute_variant.value_string
                    WHEN 'number' THEN attribute_variant.value_integer
                    WHEN 'wysiwyg' THEN {$translatedValueQuery}
                END as value
            SQL))
            ->withPivot(['value_option_id', 'value_translated', 'value_string', 'value_integer'])
            ->using(AttributeVariant::class);
    }

    public function regions()
    {
        return $this->belongsToMany(Region::class, 'region_variant', 'variant_id', 'region_id')
            ->withPivot(['product_id', 'price']);
    }

    public function loadPrice($regionId = null)
    {
        $price = $this->regions()
            ->where('region_id', $regionId ?? shop()->getRegion()->id)
            ->first();

        if (! $price) {
            return null;
        }

        $this->setAttribute('price', $price->pivot->price);

        return $price;
    }

    public function routes()
    {
        return $this->morphMany(Route::class, 'routable');
    }

    public function isOrderable($quantity = 1): bool
    {
        return $this->is_active
            && ! $this->isOutOfStock($quantity);
    }

    public function isOutOfStock($quantity = 1): bool
    {
        if (! $this->track_stock) {
            return false;
        }

        return $this->stock - $quantity < 0;
    }
}
