<?php

use FirstpointCh\Shop\Models\Address;
use FirstpointCh\Shop\Models\Customer;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Models\ShippingMethod;
use FirstpointCh\Shop\Models\TaxRule;
use FirstpointCh\Shop\Models\Variant;
use function Pest\Laravel\actingAs;

beforeEach(function () {
    $this->customer = Customer::factory()->create();
    actingAs($this->customer);

    $this->shippingMethod = ShippingMethod::factory()->create();
    $this->variant = Variant::factory()->create();

    $this->shippingMethod->regions()->attach(defaultRegion()->id, ['price' => 10]);
    $this->variant->regions()->sync([defaultRegion()->id, [
        'price' => 100,
        'product_id' => $this->variant->product_id,
    ]]);

    $this->address = Address::factory()->create(['customer_id' => $this->customer->id]);

    checkout()->update([
        'address_id' => $this->address->id,
        'first_name' => $this->customer->first_name,
        'last_name' => $this->customer->last_name,
        'company' => $this->address->company,
        'street' => $this->address->street,
        'street_number' => $this->address->street_number,
        'zip' => $this->address->zip,
        'city' => $this->address->city,
        'country' => $this->address->country,
        'shipping_method' => $this->shippingMethod->slug,
        'payment_method' => 'testing',
        'order_id' => null,
    ]);

    $this->checkout = checkout();
});

it('creates an order', function () {
    $order = checkout()->placeOrder();

    $this->assertInstanceOf(Order::class, $order);
    $this->assertEquals($order->id, checkout()->data->order_id);
});

it('updates the existing pending order', function () {
    $before = checkout()->placeOrder();

    checkout()->update([
        'first_name' => 'first_name',
        'last_name' => 'last_name',
        'company' => 'company',
        'street' => 'street',
        'street_number' => 'street_number',
        'zip' => 'zip',
        'city' => 'city',
        'country' => 'FR',
    ]);

    $after = checkout()->placeOrder();

    $this->assertEquals($before->id, $after->id); // Same order
    $this->assertEquals('first_name', $after->first_name);
    $this->assertEquals('last_name', $after->last_name);
    $this->assertEquals('company', $after->company);
    $this->assertEquals('street', $after->street);
    $this->assertEquals('street_number', $after->street_number);
    $this->assertEquals('zip', $after->zip);
    $this->assertEquals('city', $after->city);
    $this->assertEquals('FR', $after->country);
});

it('creates a new order if checkout has expired', function () {
    $first = checkout()->placeOrder();
    checkout()->update(['order_id' => null]);
    $second = checkout()->placeOrder();
    $this->assertNotEquals($first->id, $second->id);
});

it('saves taxes applied to the order', function () {
    $tax1 = TaxRule::factory()->create(['rate' => 10, 'tax_included' => false]);
    $tax2 = TaxRule::factory()->create(['rate' => 20, 'tax_included' => true]);

    $this->assertDatabaseCount('order_taxes', 0);

    cart()->add(id: $this->variant->id, quantity: 4);

    checkout()->placeOrder();

    $this->assertDatabaseCount('order_taxes', 2);
    $this->assertDatabaseHas('order_taxes', [
        'name' => $tax1->name,
        'tax_included' => false,
        'taxable_amount' => 400,
        'rate' => 10,
        'tax_amount' => 40,
    ]);
    $this->assertDatabaseHas('order_taxes', [
        'name' => $tax2->name,
        'tax_included' => true,
        'taxable_amount' => 400,
        'rate' => 20,
        'tax_amount' => 80,
    ]);
});
