<?php

use FirstpointCh\Shop\Models\Region;
use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the shipping method edit form', function () {
    $shippingMethod = ShippingMethod::factory()->create();

    $this->get('/cp/shipping-methods/'.$shippingMethod->id.'/edit')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ShippingMethod/Form')
            ->where('shippingMethod.id', $shippingMethod->id)
            ->has('regions', 1)
        );
});

test('the name is required and must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods/'.$id, ['name' => 'test'])->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods/'.$id, ['name' => ['en' => 'test']])->assertSessionDoesntHaveErrors('name');
});

test('the description is required and must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods/'.$id, ['description' => 'test'])->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods/'.$id, ['description' => ['en' => 'test']])->assertSessionDoesntHaveErrors('description');
});

test('the slug is required', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('slug');
});

test('regions must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id, ['regions' => 'test'])->assertStatus(302)->assertSessionHasErrors('regions');
    $this->post('/cp/shipping-methods/'.$id, ['regions' => []])->assertStatus(302)->assertSessionDoesntHaveErrors('regions');
});

test('each region must exist and have a price', function () {
    $region = Region::factory()->create();
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id, [
        'regions' => [
            ['id' => 123],
        ],
    ])->assertStatus(302)->assertSessionHasErrors('regions.0.id');

    $this->post('/cp/shipping-methods', [
        'regions' => [
            ['id' => $region->id],
        ],
    ])->assertStatus(302)->assertSessionDoesntHaveErrors('regions.0.id');
});

it('updates a shipping method', function () {
    $shippingMethod = ShippingMethod::factory()->create();

    $data = [
        'name' => ['en' => 'My shipping method'],
        'description' => ['en' => 'My shipping method'],
        'slug' => 'my-shipping-method',
        'regions' => [
            ['id' => defaultRegion()->id, 'price' => 123],
        ],
    ];

    $this->assertDatabaseCount('region_shipping_method', 0);

    $this->post('/cp/shipping-methods/'.$shippingMethod->id, $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/shipping-methods');

    $regions = Arr::pull($data, 'regions');
    $data['name'] = json_encode($data['name']);
    $data['description'] = json_encode($data['description']);
    $data['id'] = $shippingMethod->id;

    $this->assertDatabaseCount('region_shipping_method', 1);
    $this->assertDatabaseHas('shipping_methods', $data);
    $this->assertDatabaseHas('region_shipping_method', [
        'region_id' => $regions[0],
        'shipping_method_id' => ShippingMethod::first()->id,
        'price' => 123,
    ]);
});
