<?php

namespace FirstpointCh\Shop\Payment\Gateway;

use FirstpointCh\Shop\Exceptions\PaymentDoesNotExist;
use FirstpointCh\Shop\Models\Payment;
use Illuminate\Http\Request;

class StripeCheckoutGateway extends PaymentGateway
{
    public $mode = 'redirect';

    private $webhookEvent;

    public function init(Payment $payment)
    {
        $lineItems = $this->payment->order->items->map(function ($item) {
            return [
                'price_data' => [
                    // 'currency' => 'chf',
                    'product_data' => [
                        'name' => $item->name,
                    ],
                    'unit_amount' => (int) ($item->unit_price * 100),
                ],
                'quantity' => $item->quantity,
            ];
        });

        $this->session = \Stripe\Checkout\Session::create([
            'client_reference_id' => $this->payment->reference,
            'line_items' => $lineItems,
            'payment_method_types' => [
                'card',
            ],
            'mode' => 'payment',
            'success_url' => url('/payment/stripe-checkout/success'),
            'cancel_url' => url('/payment/stripe-checkout/cancel'),
        ]);
    }

    public function redirect()
    {
        return $this->session->url;
    }

    public static function fromRequest(Request $request)
    {
        $payload = @file_get_contents('php://input');

        try {
            $this->webhookEvent = \Stripe\Webhook::constructEvent(
                $payload,
                $_SERVER['HTTP_STRIPE_SIGNATURE'],
                $endpoint_secret
            );
        } catch (\UnexpectedValueException $e) {
            abort(400, 'Invalid payload');
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            abort(400, 'Invalid signature');
        }

        $payment = Payment::where('reference', $this->webhookEvent->data->object->client_reference_id)
            ->first();

        if (empty($payment)) {
            throw new PaymentDoesNotExist();
        }

        return new self($payment);
    }

    public function validate(Request $request)
    {
        return $this->webhookEvent->type === 'checkout.session.completed';
    }

    public function onSuccess()
    {
        // Do something...
    }
}
