<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Region;
use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Inertia;

class ShippingMethodController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('ShippingMethod/Index', [
            'shippingMethods' => $datatable
                ->searchable(['name', 'description', 'slug'])
                ->performQuery(
                    ShippingMethod::query()
                ),
            'datatableState' => $datatable->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('ShippingMethod/Form', [
            'regions' => Region::all(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'regions' => 'nullable|array',
            'regions.*.id' => 'required|exists:regions,id',
            'regions.*.price' => 'nullable',
        ]);

        $regions = collect(Arr::pull($data, 'regions'))
            ->keyBy('id')
            ->map(fn ($item) => ['price' => $item['price']])
            ->toArray();

        $shippingMethod = ShippingMethod::create($data);

        $shippingMethod->regions()->attach($regions);

        return redirect('/cp/shipping-methods');
    }

    public function edit(ShippingMethod $shippingMethod)
    {
        return Inertia::render('ShippingMethod/Form', [
            'shippingMethod' => $shippingMethod->in('*')->load('regions'),
            'regions' => Region::all(),
        ]);
    }

    public function update(ShippingMethod $shippingMethod)
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'regions' => 'nullable|array',
            'regions.*.id' => 'required|exists:regions,id',
            'regions.*.price' => 'nullable',
        ]);

        $regions = collect(Arr::pull($data, 'regions'))
            ->keyBy('id')
            ->map(fn ($item) => ['price' => $item['price']])
            ->toArray();

        $shippingMethod->update($data);

        $shippingMethod->regions()->sync($regions);

        return redirect('/cp/shipping-methods');
    }
}
