<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Customer;
use Inertia\Inertia;

class CustomerController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Customer/Index', [
            'customers' => $datatable
                ->searchable(['first_name', 'last_name', 'email'])
                ->performQuery(
                    Customer::query()
                ),
            'datatableState' => $datatable->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('Customer/Form');
    }

    public function store()
    {
        Customer::create(request()->validate([
            'first_name' => 'required|max:255',
            'last_name' => 'required|max:255',
            'email' => 'required|max:255|email|unique:customers,email',
            'password' => 'required',
        ]));

        return redirect('/cp/customers');
    }

    public function show(Customer $customer, string $tab = null)
    {
        return Inertia::render('Customer/Show', [
            'tab' => $tab,
            'customer' => $customer,
            'addresses' => $customer->addresses()->orderBy('id', 'desc')->paginate(10),
            'carts' => $customer->carts()->with('region')->paginate(10),
            'orders' => $customer->orders()->orderBy('id', 'desc')->paginate(10),
            'emails' => $customer->notificationLogs()->orderBy('id', 'desc')->paginate(10),
        ]);
    }

    public function edit(Customer $customer)
    {
        return Inertia::render('Customer/Form', [
            'customer' => $customer,
        ]);
    }

    public function update(Customer $customer)
    {
        $data = request()->validate([
            'first_name' => 'required|max:255',
            'last_name' => 'required|max:255',
            'email' => 'required|max:255|email|unique:customers,email,'.$customer->id,
            'password' => 'nullable',
        ]);

        if (empty($data['password'])) {
            unset($data['password']);
        } else {
            $data['password'] = bcrypt($data['password']);
        }

        $customer->update($data);

        return redirect('/cp/customers');
    }
}
