<?php

namespace FirstpointCh\Shop\Catalog\QueryBuilders;

use FirstpointCh\Shop\Models\Attribute;
use Illuminate\Database\Eloquent\Builder;

class ProductQueryBuilder extends Builder
{
    public function published()
    {
        return $this->where('status', 'published');
    }

    public function fromCategories($categories = [])
    {
        if (empty($categories)) {
            return $this;
        }

        return $this->whereHas('categories', function ($query) use ($categories) {
            $query->whereIn('categories.id', $categories);
        });
    }

    // TODO: Remove if not used
    public function wherePriceBetween($min = null, $max = null)
    {
        return $this->whereHas('regions', function ($query) use ($min, $max) {
            $query
                ->when(shop()->isRegionAware(), fn ($query) => $query->where('region_id', shop()->getRegion()->id))
                ->when($min, fn ($query) => $query->where('price', '>=', $min))
                ->when($max, fn ($query) => $query->where('price', '<=', $max));
        });
    }

    public function whereAttributeIn(string $type, string $attribute, array $values, ?string $locale = null)
    {
        return $this->whereHas('attributes', function ($query) use ($type, $attribute, $values, $locale) {
            $query->where('attributes.key', $attribute);

            foreach ($values as $value) {
                if ($type === 'text') {
                    $query->orWhere('attribute_product.value_string', $value);
                } elseif ($type === 'translated') {
                    $query->orWhere("attribute_product.value_translated->{$locale}", $value);
                } elseif ($type === 'integer') {
                    $query->orWhere('attribute_product.value_integer', $value);
                } elseif ($type === 'options') {
                    $query->orWhere('attribute_product.value_option_id', $value);
                }
            }
        });
    }

    public function withTotalTaxRate($type = null)
    {
        return $this->addSelect(['total_tax_rate' => function ($query) use ($type) {
            $query->selectRaw('SUM(tax_rules.rate)')
                ->from('tax_rules')
                ->when($type === 'included', function ($query) {
                    $query->where('tax_rules.tax_included', true);
                })
                ->when($type === 'excluded', function ($query) {
                    $query->where('tax_rules.tax_included', false);
                })
                ->where(function ($query) {
                    // The tax rule must be applied to all regions or the current region
                    $query->where('tax_rules.apply_to_all_regions', true)
                        ->orWhereExists(function ($query) {
                            $query->selectRaw(1)
                                ->from('region_tax_rule')
                                ->whereColumn('region_tax_rule.tax_rule_id', 'tax_rules.id')
                                ->where('region_tax_rule.region_id', 1);
                        });
                })
                ->where(function ($query) {
                    // The tax rule must be applied to all product types or this product type
                    $query->where('tax_rules.apply_to_all_product_types', true)
                        ->orWhereExists(function ($query) {
                            $query->selectRaw(1)
                                ->from('product_type_tax_rule')
                                ->whereColumn('product_type_tax_rule.tax_rule_id', 'tax_rules.id')
                                ->whereColumn('product_type_tax_rule.product_type_id', 'products.product_type_id');
                        });
                });
        }])->withCasts([
            'total_tax_rate' => 'float',
        ]);
    }

    public function whereAttribute(Attribute $attribute, $value)
    {
        if ($attribute->type === 'text') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_product.value_string', $value);
            });
        }

        if ($attribute->type === 'translated') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_product.value_translated->'.app()->getLocale(), $value);
            });
        }

        if ($attribute->type === 'integer') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_product.value_integer', $value);
            });
        }

        if ($attribute->type === 'options') {
            return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
                $query->where('attributes.key', $attribute->key)
                    ->where('attribute_product.value_option_id', $value);
            });
        }
    }
}
