<?php

namespace FirstpointCh\Shop\Payment\Actions;

use FirstpointCh\Shop\Models\Cart;
use FirstpointCh\Shop\Actions\Action;
use FirstpointCh\Shop\Models\Order;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Models\ShippingMethod;
use FirstpointCh\Shop\Order\Enums\OrderStatus;
use FirstpointCh\Shop\Order\Enums\OrderItemStatus;

class PaymentIntentSuccessAction
{
    use Action;

    public function handle(Payment $payment)
    {
        return DB::transaction(function () use ($payment) {
            // Create order
            $order = Order::create([
                'customer_id' => $payment->cart->customer_id,
                'first_name' => $payment->cart->first_name,
                'last_name' => $payment->cart->last_name,
                'email' => $payment->cart->customer->email,
                'company' => $payment->cart->company,
                'street' => $payment->cart->street,
                'street_number' => $payment->cart->street_number,
                'zip' => $payment->cart->zip,
                'city' => $payment->cart->city,
                'country' => $payment->cart->country,
                'use_billing_address' => false,
                'shipping_price' => ShippingMethod::with('regions')
                    ->firstWhere('slug', $payment->cart->shipping_method)
                    ->regions
                    ->where('id', $payment->cart->region_id)
                    ->first()
                    ->pivot
                    ->price,
                'amount' => $payment->amount,
                'currency' => $payment->currency,
                'status' => OrderStatus::Pending,
                'payment_method' => $payment->cart->payment_method,
                'shipping_method' => $payment->cart->shipping_method,
                'region_id' => $payment->cart->region_id,
            ]);

            // Create order items
            $payment->cart->items->each(function ($item) use ($order) {
                $order->items()->create([
                    'sku' => $item->orderable->sku,
                    'name' => $item->name,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->orderable->price,
                    'total_tax_rate' => $item->orderable->product->total_tax_rate, // TODO: Test
                    'status' => OrderItemStatus::Pending,
                ]);
            });

            cart()->taxes->each(fn ($tax) => $order->taxes()->create([
                'name' => $tax->taxRule->name,
                'tax_included' => $tax->taxRule->tax_included,
                'taxable_amount' => $tax->taxable_amount,
                'rate' => $tax->taxRule->rate,
                'tax_amount' => $tax->taxRule->calculate($tax->taxable_amount),
            ]));

            // Update cart
            $payment->cart->update([
                'order_id' => $order->id,
            ]);

            // Update payment
            $payment->update([
                'order_id' => $order->id,
                'status' => 'paid',
            ]);

            return $order;
        });
    }
}
