<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Data\Cp\DefaultVariantAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductVariantFormData;
use FirstpointCh\Shop\Data\Cp\VariantAttributeData;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\CustomField;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\Region;
use FirstpointCh\Shop\Models\TaxRule;
use FirstpointCh\Shop\Models\Variant;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ProductVariantController extends Controller
{
    public function create(Product $product)
    {
        $variant = new Variant([
            'is_active' => false,
            'name' => [config('app.fallback_locale') => null],
            'sku' => null,
            'slug' => [config('app.fallback_locale') => null],
            'is_shippable' => true,
            'weight_grams' => null,
            'track_stock' => false,
            'stock' => null,
            'attributes' => [],
            'regions' => [],
            'base_attributes' => [],
            'extra_attributes' => [],
            'custom_fields' => [],
        ]);

        if (! empty($product->productType)) {
            $variant->setAttribute('base_attributes', DefaultVariantAttributeData::collection(
                $product->productType->variantAttributes
            ));
        }

        return Inertia::render('ProductVariant/Form', [
            'product' => $product,
            'variant' => $variant->in('*'),
            'attributes' => Attribute::all(),
            'regions' => Region::all(),
            'taxRules' => TaxRule::all(),
            'customFields' => CustomField::all(),
        ]);
    }

    public function store(Product $product, ProductVariantFormData $data)
    {
        DB::transaction(function () use ($product, $data) {
            $variant = $product->variants()->create($data->getVariantData());
            $variant->attributes()->sync($data->getAttributes());
            $variant->regions()->sync($data->getRegions($product->id));
            $variant->updateCustomFields($data->custom_fields);
            $variant->index();
        });

        return redirect()->route('cp.products.show', [
            'product' => $product,
            'tab' => 'variants',
        ])->with('toast', [
            'title' => 'Variante créé',
            'message' => 'La variante a bien été créé',
            'type' => 'success',
        ]);
    }

    public function edit(Product $product, Variant $variant)
    {
        $baseAttributeIds = $product->productType
            ? $product->productType->variantAttributes->pluck('id')->toArray()
            : [];

        if (! empty($product->productType)) {
            $baseAttributes = $variant->attributes()->whereIn('attributes.id', $baseAttributeIds)->get();

            $variant->setAttribute('base_attributes', $product->productType->variantAttributes->map(fn ($attribute) => [
                'attribute_id' => $attribute->id,
                'name' => $attribute->name,
                'type' => $attribute->type,
                'translatable' => $attribute->translatable,
                'options' => $attribute->options->map->in('*'),
                'value' => $attribute->translatable
                    ? $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value->raw() ?? [config('app.fallback_locale') => null]
                    : $baseAttributes->firstWhere('id', $attribute->id)?->pivot->value->raw(),
            ]));
        }

        $variant->setAttribute('extra_attributes', VariantAttributeData::collection(
            $variant->attributes()->whereNotIn('attributes.id', $baseAttributeIds)->get()->map(fn ($attribute) => [
                'attribute_id' => $attribute->id,
                'name' => $attribute->name,
                'type' => $attribute->type,
                'translatable' => $attribute->translatable,
                'options' => $attribute->options->map->in('*'),
                'value' => $attribute->pivot->value->raw(),
            ])
        ));

        return Inertia::render('ProductVariant/Form', [
            'product' => $product,
            'variant' => $variant->load(['regions', 'customFields'])->in('*'),
            'attributes' => Attribute::orderBy('name->'.config('app.fallback_locale'))->get(),
            'regions' => Region::all(),
            'taxRules' => TaxRule::all(),
            'customFields' => CustomField::all(),
        ]);
    }

    public function update(Product $product, Variant $variant, ProductVariantFormData $data)
    {
        DB::transaction(function () use ($variant, $data) {
            $variant->update($data->getVariantData());
            $variant->attributes()->sync($data->getAttributes());
            $variant->regions()->sync($data->getRegions($variant->product_id));
            $variant->updateCustomFields($data->custom_fields);
            $variant->index();
        });

        return redirect()->route('cp.products.show', [
            'product' => $product,
            'tab' => 'variants',
        ])->with('toast', [
            'title' => 'Variante enregistrée',
            'message' => 'La variante a bien été enregistrée',
            'type' => 'success',
        ]);
    }

    public function delete(Product $product, Variant $variant)
    {
        $variant->delete();
        $variant->index();

        return redirect('/cp/products/'.$product->id.'/variants');
    }

    public function forceDelete(Product $product, $variant)
    {
        Variant::withTrashed()->find($variant)->forceDelete();

        return redirect('/cp/products/'.$product->id.'/variants');
    }

    public function restore(Product $product, $variant)
    {
        $variant = Variant::withTrashed()->find($variant);

        $variant->restore();
        $variant->index();

        return redirect('/cp/products/'.$product->id.'/variants');
    }
}
