<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use DB;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Region;
use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Inertia;

class ShippingMethodController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('ShippingMethod/Index', [
            'shippingMethods' => $datatable
                ->searchable(['name', 'description', 'slug'])
                ->query(ShippingMethod::query())
                ->get(),
            'datatableState' => $datatable->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('ShippingMethod/Form', [
            'regions' => Region::all(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'regions' => 'nullable|array',
            'regions.*.region_id' => 'required|exists:regions,id',
            'regions.*.min_price' => 'nullable',
            'regions.*.max_price' => 'nullable',
            'regions.*.price' => 'required',
        ]);

        $regions = Arr::pull($data, 'regions');

        ShippingMethod::create($data)
            ->regions()
            ->attach($regions);

        return redirect('/cp/shipping-methods');
    }

    public function edit(ShippingMethod $shippingMethod)
    {
        return Inertia::render('ShippingMethod/Form', [
            'shippingMethod' => $shippingMethod->in('*')->load('regions'),
            'regions' => Region::all(),
        ]);
    }

    public function update(ShippingMethod $shippingMethod)
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'regions' => 'nullable|array',
            'regions.*.pivot_id' => 'nullable|exists:region_shipping_method,id',
            'regions.*.region_id' => 'required|exists:regions,id',
            'regions.*.min_price' => 'nullable',
            'regions.*.max_price' => 'nullable',
            'regions.*.price' => 'required',
        ]);

        $regions = Arr::pull($data, 'regions');

        DB::transaction(function () use ($data, $regions, $shippingMethod) {
            $shippingMethod->update($data);

            $ids = Arr::whereNotNull(Arr::pluck($regions, 'pivot_id'));

            DB::table('region_shipping_method')->whereNotIn('id', $ids)->delete();

            foreach ($regions as $region) {
                if (! empty($region['pivot_id'])) {
                    DB::table('region_shipping_method')->where('id', $region['pivot_id'])->update([
                        'min_price' => $region['min_price'],
                        'max_price' => $region['max_price'],
                        'price' => $region['price'],
                    ]);
                } else {
                    $shippingMethod->regions()->attach($region['region_id'], [
                        'min_price' => $region['min_price'],
                        'max_price' => $region['max_price'],
                        'price' => $region['price'],
                    ]);
                }
            }
        });

        return redirect('/cp/shipping-methods');
    }
}
