<?php

namespace FirstpointCh\Shop\Data\Cp;

use FirstpointCh\Shop\Models\Attribute;
use Spatie\LaravelData\Data;

class ProductVariantFormData extends Data
{
    public function __construct(
        public ?bool $is_active,
        public array $name,
        public string $sku,
        public array $slug,
        public bool $track_stock,
        public ?int $stock,
        public bool $is_shippable = true,
        public ?float $weight_grams = null,
        public ?array $base_attributes = [],
        public ?array $extra_attributes = [],
        public ?array $regions = [],
        public ?array $custom_fields = [],
    ) {
    }

    public static function rules(): array
    {
        return [
            'is_active' => 'nullable',
            'name' => 'required',
            'sku' => 'required',
            'slug' => 'required',
            'track_stock' => 'required|boolean',
            'stock' => 'nullable|numeric',
            'is_shippable' => 'boolean',
            'weight_grams' => 'nullable|numeric',
            'base_attributes' => 'array',
            'base_attributes.*.attribute_id' => 'required|exists:attributes,id',
            'base_attributes.*.value' => 'nullable',
            'extra_attributes' => 'array',
            'extra_attributes.*.attribute_id' => 'required|exists:attributes,id',
            'extra_attributes.*.value' => 'nullable',
            'regions' => 'array',
            'regions.*.id' => 'required|exists:regions,id',
            'regions.*.is_active' => 'nullable|boolean',
            'regions.*.price' => 'required_if:regions.*.is_active,1',
            'custom_fields' => 'nullable|array',
        ];
    }

    public function getVariantData()
    {
        return [
            'is_active' => $this->is_active,
            'name' => $this->name,
            'sku' => $this->sku,
            'track_stock' => $this->track_stock,
            'stock' => $this->stock,
            'slug' => $this->slug,
            'is_shippable' => $this->is_shippable,
            'weight_grams' => $this->weight_grams,
        ];
    }

    public function getAttributes()
    {
        $baseAttributes = collect($this->base_attributes)
            ->mapWithKeys(function ($attr) {
                $attribute = Attribute::find($attr['attribute_id']);

                return [
                    $attr['attribute_id'] => [
                        'value' => $attr['value'],
                        'type' => $attribute->type,
                        'translatable' => $attribute->translatable,
                    ],
                ];
            })->toArray();

        $extraAttributes = collect($this->extra_attributes)
            ->mapWithKeys(function ($attr) {
                $attribute = Attribute::find($attr['attribute_id']);

                return [
                    $attr['attribute_id'] => [
                        'value' => $attr['value'],
                        'type' => $attribute->type,
                        'translatable' => $attribute->translatable,
                    ],
                ];
            })->toArray();

        return $baseAttributes + $extraAttributes; // Merge but keep keys
    }

    public function getRegions($productId)
    {
        return collect($this->regions)
            ->where('is_active', true)
            ->keyBy('id')
            ->map(fn ($region) => [
                'product_id' => $productId,
                'price' => $region['price'],
            ]);
    }
}
