<?php

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductType;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the product edit form', function () {
    $productType = ProductType::factory()->create();
    $product = Product::factory()->create([
        'product_type_id' => $productType->id,
    ]);

    $this->get('/cp/products/'.$product->id.'/edit')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('Product/Form')
            ->has('productTypes', 1)
            ->has('product.base_attributes', 0)  // No attributes on the product type
            ->whereAll([
                'productTypes.0.id' => $productType->id,
                'product.id' => $product->id,
                // TODO: Add all properties ?
            ])
        );
});

it('has default attributes based on product type', function () {
    $productType = ProductType::factory()->create();
    $attribute = Attribute::factory()->create();

    Attribute::factory()->create(); // This attribute should not be added by default

    $productType->productAttributes()->attach($attribute->id, ['type' => 'product']);

    $product = Product::factory()->create([
        'product_type_id' => $productType->id,
    ]);

    $this->get('/cp/products/'.$product->id.'/edit')
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('Product/Form')
            ->has('product.base_attributes', 1)
            ->whereAll([
                'product.base_attributes.0.attribute_id' => $attribute->id,
                'product.base_attributes.0.name' => $attribute->name,
            ])
        );
});

test('name is required', function () {
    $product = Product::factory()->create();

    $this->post('/cp/products/'.$product->id)
        ->assertStatus(302)
        ->assertSessionHasErrors('name');
});

test('product_type_id must exist', function () {
    $product = Product::factory()->create();

    $this->post('/cp/products/'.$product->id, ['product_type_id' => 123])
        ->assertStatus(302)
        ->assertSessionHasErrors('product_type_id');

    $this->post('/cp/products/'.$product->id, ['product_type_id' => ProductType::factory()->create()->id])
        ->assertStatus(302)
        ->assertSessionDoesntHaveErrors('product_type_id');
});

test('each base_attribute must exist', function () {
    $product = Product::factory()->create();
    $attribute = Attribute::factory()->create();

    $this->post('/cp/products/'.$product->id, [
        'base_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['base_attributes.0.attribute_id'])
    ->assertSessionHasErrors(['base_attributes.1.attribute_id']);
});

test('each extra_attribute must exist', function () {
    $product = Product::factory()->create();
    $attribute = Attribute::factory()->create();

    $this->post('/cp/products/'.$product->id, [
        'extra_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['extra_attributes.0.attribute_id'])
    ->assertSessionHasErrors(['extra_attributes.1.attribute_id']);
});

it('updates a product', function () {
    $product = Product::factory()->create();
    $baseAttribute = Attribute::factory()->create([
        'type' => 'text',
    ]);
    $extraAttribute = Attribute::factory()->create([
        'type' => 'text',
        'translatable' => true,
    ]);

    $data = [
        'status' => 'draft',
        'name' => ['en' => 'Product name'],
        'short_description' => ['en' => 'short description'],
        'description' => ['en' => 'description'],
        'product_type_id' => ProductType::factory()->create()->id,
        'base_attributes' => [
            ['attribute_id' => $baseAttribute->id, 'value' => 'test'],
        ],
        'extra_attributes' => [
            ['attribute_id' => $extraAttribute->id, 'value' => ['en' => 'test extra']],
        ],
    ];

    $this->post('/cp/products/'.$product->id, $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/products/'.$product->id);

    $data['name'] = json_encode($data['name']);
    $data['short_description'] = json_encode($data['short_description']);
    $data['description'] = json_encode($data['description']);

    Arr::pull($data, 'base_attributes');
    Arr::pull($data, 'extra_attributes');

    $this->assertDatabaseHas('products', $data);
    $this->assertCount(2, $product->attributes);
    $this->assertEquals('test', $product->attributes->first()->pivot->value);
    $this->assertEquals('test extra', $product->attributes->last()->pivot->value);
});
