<?php

namespace FirstpointCh\Shop;

use FirstpointCh\Shop\Cart\CartManager;
use FirstpointCh\Shop\Commands\CatalogIndex;
use FirstpointCh\Shop\Commands\InstallShop;
use FirstpointCh\Shop\Commands\PublishAssets;
use FirstpointCh\Shop\Commands\RebuildCategoryPaths;
use FirstpointCh\Shop\Listeners\LogSentNotification;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Observers\OrderObserver;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Str;

class ShopServiceProvider extends ServiceProvider
{
    public function register()
    {
        $this->app->singleton('shop::shop', function () {
            return new Shop(
                session('shop::currentRegion', config('shop.default_region')),
            );
        });

        $this->app->singleton('shop::money', function () {
            return new Money(region()->currency, region()->currency_format);
        });

        $this->app->singleton('shop::cart', function () {
            return new CartManager();
        });

        $this->app->singleton(\FirstpointCh\Shop\Payment\Gateway::class, function () {
            return new \FirstpointCh\Shop\Payment\Gateway(config('shop.payment_gateways'));
        });

        $this->app->bind('shop::carriers', function () {
            return new \FirstpointCh\Shop\Carrier\CarrierManager(config('shop.carriers'));
        });
    }

    public function boot()
    {
        $this->configureRoutes();
        $this->registerRoutes();

        $this->loadViewsFrom(__DIR__.'/../resources/views', 'shop');
        $this->loadViewsFrom(__DIR__.'/../resources/views/invoices', 'invoices');

        $this->loadTranslationsFrom(__DIR__.'/../resources/lang', 'shop');

        $this->publishes([
            __DIR__.'/../config/shop.php' => config_path('shop.php'),
            __DIR__.'/../database/migrations/' => database_path('migrations'),
        ], 'shop');

        $this->publishes([
            __DIR__.'/../public/build' => public_path('vendor/shop'),
        ], 'shop-assets');

        $this->configureCpGuard();

        if ($this->app->runningInConsole()) {
            $this->commands([
                InstallShop::class,
                CatalogIndex::class,
                PublishAssets::class,
                RebuildCategoryPaths::class,
            ]);
        }

        Order::observe(OrderObserver::class);

        Event::listen(
            [\Illuminate\Notifications\Events\NotificationSent::class],
            LogSentNotification::class
        );
    }

    public function configureRoutes()
    {
        Route::macro('localized', function ($uri, $action, $name, $methods = ['GET']) {
            $uri = Str::startsWith($uri, '/') ? $uri : '/'.$uri;

            foreach (config('shop.locales', []) as $locale => $config) {
                if ($locale === config('app.fallback_locale')) {
                    Route::match($methods, $uri, $action)->name("$locale.$name");

                    continue;
                }

                if (Lang::has("routes.{$uri}", $locale)) {
                    Route::match($methods, $config['base_path'].Lang::get("routes.{$uri}", [], $locale), $action)->name("$locale.$name");
                } else {
                    Route::match($methods, $config['base_path'].$uri, $action)->name("$locale.$name");
                }
            }
        });
    }

    public function registerRoutes()
    {
        if (config('shop.api.enabled')) {
            $this->loadRoutesFrom(__DIR__.'/../routes/api.php');
        }

        $this->loadRoutesFrom(__DIR__.'/../routes/cp.php');
        $this->loadRoutesFrom(__DIR__.'/../routes/web.php');
    }

    public function configureCpGuard()
    {
        $this->app['config']->set('auth.guards.'.config('shop.cp.guard'), [
            'driver' => 'session',
            'provider' => 'cp_users',
        ]);

        $this->app['config']->set('auth.providers.cp_users', [
            'driver' => 'eloquent',
            'model' => \FirstpointCh\Shop\Models\User::class,
        ]);
    }
}
