<?php

namespace FirstpointCh\Shop\Payment\Gateway;

use FirstpointCh\Shop\Models\Cart;
use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Payment\Enums\GatewayStatus;
use Illuminate\Http\Request;

abstract class PaymentGateway
{
    /**
     * The slug of the gateway. This is a slug used in the URL.
     */
    public string $slug;

    /**
     * The name of the gateway.
     */
    public string $name;

    /**
     * The description of the gateway.
     */
    public string $description;

    /**
     * The configuration
     */
    public array $config = [];

    /**
     * Online: The payment is handled in a stateful request
     * Offline: The payment is handled througt a stateless server-to-server request (ex: webhook)
     */
    public string $handleMode = 'online';

    public Payment $payment;

    public function __construct(array $config = [])
    {
        $this->config = $config;
    }

    public function init(Payment $payment): self
    {
        $this->payment = $payment;

        return $this;
    }

    /**
     * This method should set the payment property based on an input request
     *
     * @param  Request  $request
     * @return self
     */
    abstract public function fromRequest(Request $request): self;

    /**
     * Returns a boolean indicating if the gateway is enabled for the given cart
     */
    public function enabled(Cart $cart): bool
    {
        return true;
    }

    /**
     * Returns the fee for the given cart
     */
    public function calculateFee(Cart $cart): float
    {
        return 0;
    }

    /**
     * This method must redirect the customer to a payment platform or a payment form
     *
     * @return void
     */
    abstract public function redirect(?string $cancelUrl = null);

    /**
     * This method validates the payment and return a GatewayStatus
     *
     * @param  Request  $request
     * @return GatewayStatus
     */
    abstract public function handle(Request $request): GatewayStatus;
}
