<?php

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductType;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the product create form', function () {
    $productType = ProductType::factory()->create();

    $this->get('/cp/products/create')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('Product/Form')
            ->has('productTypes', 1)
            ->has('product.base_attributes', 0) // No attributes on the product type
            ->where('productTypes.0.id', $productType->id)
        );
});

it('has default attributes based on product type', function () {
    $productType = ProductType::factory()->create();
    $attribute = Attribute::factory()->create();

    Attribute::factory()->create(); // This attribute should not be added by default

    $productType->productAttributes()->attach($attribute->id, ['type' => 'product']);

    $this->get('/cp/products/create?productTypeId='.$productType->id)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('Product/Form')
            ->has('product.base_attributes', 1)
            ->whereAll([
                'product.base_attributes.0.attribute_id' => $attribute->id,
                'product.base_attributes.0.name' => $attribute->name,
            ])
        );
});

test('name is required', function () {
    $this->post('/cp/products', ['name' => null])->assertSessionHasErrors('name');
    $this->post('/cp/products', ['name' => ['en' => 'name']])->assertSessionDoesntHaveErrors('name');
});

test('product_type_id must exist', function () {
    $this->post('/cp/products', ['product_type_id' => 123])
        ->assertStatus(302)
        ->assertSessionHasErrors('product_type_id');

    $this->post('/cp/products', ['product_type_id' => ProductType::factory()->create()->id])
        ->assertStatus(302)
        ->assertSessionDoesntHaveErrors('product_type_id');
});

it('creates a product', function () {
    $baseAttribute = Attribute::factory()->create([
        'type' => 'text',
    ]);

    $extraAttribute = Attribute::factory()->create([
        'type' => 'number',
    ]);

    $data = [
        'name' => ['en' => 'Product name'],
        'short_description' => null,
        'description' => null,
        'product_type_id' => ProductType::factory()->create()->id,
        'base_attributes' => [
            ['attribute_id' => $baseAttribute->id, 'value_string' => 'test'],
        ],
        'extra_attributes' => [
            ['attribute_id' => $extraAttribute->id, 'value_integer' => 42],
        ],
    ];

    $this->post('/cp/products', $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/products/'.Product::first()->id);

    $data['name'] = json_encode($data['name']);
    $data['short_description'] = json_encode(['en' => null]);
    $data['description'] = json_encode(['en' => null]);

    Arr::pull($data, 'base_attributes');
    Arr::pull($data, 'extra_attributes');

    $this->assertDatabaseHas('products', $data);

    $product = Product::first();
    $this->assertCount(2, $product->attributes);
    $this->assertEquals('test', $product->attributes->first()->pivot->value_string);
    $this->assertEquals(42, $product->attributes->last()->pivot->value_integer);
});

test('new product status defaults to draft', function () {
    $data = [
        'name' => ['en' => 'Product name'],
        'short_description' => null,
        'description' => null,
        'product_type_id' => ProductType::factory()->create()->id,
    ];

    $this->post('/cp/products', $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/products/'.Product::first()->id);

    $this->assertEquals('draft', Product::first()->status);
});
