<?php

use FirstpointCh\Shop\Cart\CartManager;
use FirstpointCh\Shop\Models\Cart;
use FirstpointCh\Shop\Models\CartItem;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\TaxRule;
use FirstpointCh\Shop\Models\Variant;
use Illuminate\Support\Facades\DB;

test('the cart manager is available from the service container', function () {
    $this->assertInstanceOf(CartManager::class, app('shop::cart'));
});

test('the cart is available through a helper function', function () {
    $this->assertEquals(app('shop::cart')->cart, cart());
});

it('has many cart items', function () {
    $cart = Cart::factory()->create();
    $items = CartItem::factory()->count(3)->create([
        'cart_id' => $cart->id,
    ]);

    $this->assertInstanceOf(Illuminate\Database\Eloquent\Collection::class, $cart->items);
    $this->assertCount(3, $cart->items);
    $this->assertInstanceOf(CartItem::class, $cart->items->first());
});

it('has many taxes', function () {
    $cart = Cart::factory()->create();
    $cartTax = CartTax::factory()->create([
        'cart_id' => $cart->id,
    ]);

    $this->assertInstanceOf(Illuminate\Database\Eloquent\Collection::class, $cart->taxes);
    $this->assertCount(1, $cart->taxes);
    $this->assertInstanceOf(CartTax::class, $cart->taxes->first());
});

it('adds an items', function () {
    $cart = Cart::factory()->create();
    $variant = Variant::factory()->create();

    $variant->prices()->create([
        'region' => $cart->region,
        'price' => 100,
        'product_id' => $variant->product_id,
    ]);

    $this->assertCount(0, $cart->items);

    $cart->add(id: $variant->id, quantity: 5);

    $this->assertCount(1, $cart->refresh()->items);
    $this->assertSame($variant->reference, $cart->items->first()->orderable->reference);
    $this->assertEquals(5, $cart->items->first()->quantity);
});

it('increments quantity when an item already exists in the cart', function () {
    $cart = Cart::factory()->create(['region' => 'ch']);
    $variant = Variant::factory()->create();
    $variant->prices()->create([
        'region' => 'ch',
        'price' => 100,
        'product_id' => $variant->product_id,
    ]);

    $cart->add(id: $variant->id, quantity: 5);
    $cart->add(id: $variant->id, quantity: 5);

    $this->assertCount(1, $cart->refresh()->items);
    $this->assertSame($variant->reference, $cart->items->first()->orderable->reference);
    $this->assertEquals(10, $cart->items->first()->quantity);
});

it('stores the cart in the database before the first item is added', function () {
    $variant = Variant::factory()->create();
    $variant->prices()->create([
        'region' => 'ch',
        'price' => 100,
        'product_id' => $variant->product_id,
    ]);

    $this->assertEquals(0, DB::table('carts')->count());

    cart()->add($variant->id);

    $this->assertEquals(1, DB::table('carts')->count());
});

test('the orderable item must be active', function () {
    $cart = Cart::factory()->create();
    $variant = Variant::factory()->create([
        'is_active' => false,
    ]);

    $variant->prices()->create([
        'region' => $cart->region,
        'price' => 100,
        'product_id' => $variant->product_id,
    ]);

    $cart->add($variant->id);
})->throws(Exception::class, 'The product is not available.');

test('the parent product must be published', function () {
    $cart = Cart::factory()->create();
    $variant = Variant::factory()->create([
        'is_active' => true,
        'product_id' => Product::factory()->create([
            'status' => 'draft',
        ]),
    ]);

    $variant->prices()->create([
        'region' => $cart->region,
        'price' => 100,
        'product_id' => $variant->product_id,
    ]);

    $cart->add($variant->id);
})->throws(Exception::class, 'The product is not available.');

it('calculates the subtotal', function () {
    $cart = Cart::factory()->create();
    $variant = Variant::factory()->create();

    $variant->prices()->create([
        'region' => $cart->region,
        'price' => 100,
        'product_id' => $variant->product_id,
    ]);

    $cart->add(id: $variant->id, quantity: 5);

    $this->assertEquals(500, $cart->getSubtotal());
});

it('resets the cart', function () {
    /** @var Variant $variant */
    $variant = Variant::factory()->create();
    $variant->prices()->create(['region' => cart()->region, 'price' => 100, 'product_id' => $variant->product_id]);

    cart()->add(id: $variant->id, quantity: 5);

    $this->assertNotNull(cart()->id);
    $cartId = cart()->id;

    cart()->reset();

    $this->assertNull(cart()->id);

    cart()->add(id: $variant->id, quantity: 5);

    $this->assertNotEquals($cartId, cart()->id);
});

it('create cart taxes entries', function () {
    $variant = Variant::factory()->create();
    $variant->prices()->create(['region' => cart()->region, 'price' => 100, 'product_id' => $variant->product_id]);
    $cart = Cart::factory()->create();

    TaxRule::factory()->create(['name' => 'tax1', 'rate' => 10, 'tax_included' => false]);

    $this->assertDatabaseCount('cart_taxes', 0);

    $cart->add(id: $variant->id, quantity: 5);

    $this->assertDatabaseHas('cart_taxes', [
        'cart_id' => $cart->id,
        'tax_rule_id' => 1,
        'taxable_amount' => 500,
    ]);
});

it('returns all taxes applied to the cart', function () {
    $variant = Variant::factory()->create();

    TaxRule::factory()->create(['name' => 'tax1', 'rate' => 10, 'tax_included' => false]);
    TaxRule::factory()->create(['name' => 'tax2', 'rate' => 50, 'tax_included' => false]);
    TaxRule::factory()->create(['name' => 'tax3', 'rate' => 20, 'tax_included' => true]);

    $variant->prices()->create(['region' => 'ch', 'price' => 120, 'product_id' => $variant->product_id]);

    cart()->add(id: $variant->id, quantity: 2);

    $this->assertEquals([
        ['tax_rule_id' => '1', 'taxable_amount' => 240.0, 'tax_amount' => 24.0],
        ['tax_rule_id' => '2', 'taxable_amount' => 240.0, 'tax_amount' => 120.0],
        ['tax_rule_id' => '3', 'taxable_amount' => 240.0, 'tax_amount' => 48.0],
    ], cart()->taxes->map->only(['tax_rule_id', 'taxable_amount', 'tax_amount'])->toArray());
});

it('calculates the total taxes that are included in the price', function () {
    $variant = Variant::factory()->create();

    TaxRule::factory()->create(['rate' => 10, 'tax_included' => false]);
    TaxRule::factory()->create(['rate' => 50, 'tax_included' => false]);
    TaxRule::factory()->create(['rate' => 20, 'tax_included' => true]);

    $variant->prices()->create(['region' => 'ch', 'price' => 100, 'product_id' => $variant->product_id]);

    cart()->add(id: $variant->id, quantity: 1);

    $this->assertEquals(20, cart()->taxes->included()->total());
});

it('calculates the total taxes that are not included in the price', function () {
    $variant = Variant::factory()->create();

    TaxRule::factory()->create(['rate' => 10, 'tax_included' => false]);
    TaxRule::factory()->create(['rate' => 50, 'tax_included' => false]);
    TaxRule::factory()->create(['rate' => 20, 'tax_included' => true]);

    $variant->prices()->create(['region' => 'ch', 'price' => 100, 'product_id' => $variant->product_id]);

    cart()->add(id: $variant->id, quantity: 2);

    $this->assertEquals(120, cart()->taxes->excluded()->total());
});
