<?php

use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the shipping method create form', function () {
    $this->get('/cp/shipping-methods/create')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ShippingMethod/Form')
            ->has('regions', 2)
        );
});

test('the name is required and must be an array', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods', ['name' => 'test'])->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods', ['name' => ['en' => 'test']])->assertSessionDoesntHaveErrors('name');
});

test('the description is required and must be an array', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods', ['description' => 'test'])->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods', ['description' => ['en' => 'test']])->assertSessionDoesntHaveErrors('description');
});

test('the slug is required', function () {
    $this->post('/cp/shipping-methods')->assertStatus(302)->assertSessionHasErrors('slug');
});

test('prices must be an array', function () {
    $this->post('/cp/shipping-methods', ['prices' => 'test'])->assertStatus(302)->assertSessionHasErrors('prices');
    $this->post('/cp/shipping-methods', ['prices' => []])->assertStatus(302)->assertSessionDoesntHaveErrors('prices');
});

it('stores a new shipping method', function () {
    $data = [
        'name' => ['en' => 'My shipping method'],
        'description' => ['en' => 'My shipping method'],
        'slug' => 'my-shipping-method',
        'prices' => [
            ['region' => 'ch', 'min_price' => null, 'max_price' => 50, 'price' => 123],
            ['region' => 'ch', 'min_price' => 51, 'max_price' => 100, 'price' => 200],
        ],
    ];

    $this->assertDatabaseCount('shipping_methods', 0);
    $this->assertDatabaseCount('shipping_method_prices', 0);

    $this->post('/cp/shipping-methods', $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/shipping-methods');

    $prices = Arr::pull($data, 'prices');
    $data['name'] = json_encode($data['name']);
    $data['description'] = json_encode($data['description']);

    $this->assertDatabaseCount('shipping_methods', 1);
    $this->assertDatabaseCount('shipping_method_prices', 2);
    $this->assertDatabaseHas('shipping_methods', $data);
    $this->assertDatabaseHas('shipping_method_prices', [
        'region' => $prices[0],
        'shipping_method_id' => ShippingMethod::first()->id,
        'min_price' => null,
        'max_price' => 50,
        'price' => 123,
    ]);
    $this->assertDatabaseHas('shipping_method_prices', [
        'region' => $prices[0],
        'shipping_method_id' => ShippingMethod::first()->id,
        'min_price' => 51,
        'max_price' => 100,
        'price' => 200,
    ]);
});
