<?php

namespace FirstpointCh\Shop\Payment;

use FirstpointCh\Shop\Cart\CartDriver;
use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Payment\Actions\CreatePaymentIntentAction;
use FirstpointCh\Shop\Payment\Enums\PaymentStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class StripeCheckoutGateway implements PaymentGatewayInterface
{
    public string $name = 'Stripe Checkout';

    public string $description = 'Stripe';

    public string $handleMode = 'offline';

    public ?Payment $payment = null;

    public function isEnabled(CartDriver $cart): bool
    {
        return true;
    }

    public function calculateFee(CartDriver $cart): float
    {
        return 0;
    }

    public function getMode(): string
    {
        return 'redirect';
    }

    public function createPaymentIntent(CartDriver $cart): self
    {
        $this->payment = CreatePaymentIntentAction::run($cart, 'stripe-checkout');

        return $this;
    }

    public function redirect()
    {
        \Stripe\Stripe::setApiKey(config('shop.payment_gateways.drivers.stripe-checkout.secret_key'));

        $session = \Stripe\Checkout\Session::create([
            'line_items' => [[
                'price_data' => [
                    'currency' => $this->payment->currency,
                    'unit_amount' => (int) ($this->payment->amount * 100),
                    'product_data' => [
                        'name' => 'Panier #'.$this->payment->cart_id,
                    ],
                ],
                'quantity' => 1,
            ]],
            'customer_email' => customer()->email,
            'client_reference_id' => $this->payment->reference,
            'mode' => 'payment',
            'success_url' => route('shop::payment.confirmation', [
                'gateway' => 'stripe-checkout',
                'reference' => $this->payment->reference,
            ]),
            'cancel_url' => config('shop.payment-gateways.cancel_url'),
        ]);

        return redirect($session->url);
    }

    public function handle(Request $request): Payment
    {
        $data = $request->all();
        $reference = $data['data']['object']['client_reference_id'];

        $this->payment = Payment::where('reference', $reference)->firstOrFail();

        shop()->setRegion($this->payment->cart->region->slug);

        \Stripe\Stripe::setApiKey(config('shop.payment_gateways.drivers.stripe-checkout.secret_key'));

        try {
            $event = \Stripe\Webhook::constructEvent(
                @file_get_contents('php://input'),
                $request->header('Stripe-Signature'),
                config('shop.payment_gateways.drivers.stripe-checkout.webhook_secret'),
            );
        } catch (\UnexpectedValueException $e) {
            throw new \Exception('Invalid payload.');
        } catch (\Stripe\Exception\SignatureVerificationException $e) {
            throw new \Exception('Invalid signature.');
        }

        Log::info($event->type);
        Log::info($event->data->object->client_reference_id);

        $this->payment = Payment::where('reference', $event->data->object->client_reference_id)->firstOrFail();

        if ($event->type === 'checkout.session.completed' && $event->data->object->payment_status === 'paid') {
            $this->payment->update([
                'status' => PaymentStatus::Paid,
                'status_message' => null,
            ]);
        } elseif ($event->type === 'checkout.session.completed' && $event->data->object->payment_status === 'unpaid') {
            $this->payment->update([
                'status' => PaymentStatus::Failed,
                'status_message' => 'Payment failed.',
            ]);
        } else {
            $this->payment->update([
                'status' => PaymentStatus::Unknown,
                'status_message' => null,
            ]);
        }

        return $this->payment;
    }
}
