<?php

namespace FirstpointCh\Shop\Payment\Actions;

use FirstpointCh\Shop\Actions\Action;
use FirstpointCh\Shop\Models\Cart;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Observers\OrderObserver;
use FirstpointCh\Shop\Order\Enums\OrderItemStatus;
use FirstpointCh\Shop\Order\Enums\OrderStatus;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

/**
 * @method static run(Payment $payment, OrderStatus $orderStatus, Cart $cart)
 */
class CreateOrderAction
{
    use Action;

    public function handle(Payment $payment, OrderStatus $orderStatus, Cart $cart)
    {
		if (!empty($cart->order)) {
			return;
		}
		
        $order = DB::transaction(function () use ($payment, $orderStatus, $cart) {
            $order = Order::withoutEvents(function () use ($payment, $orderStatus, $cart) {
                return Order::create([
                    'reference' => Str::random(10),
                    'customer_id' => $payment->customer_id,
                    'first_name' => $cart->shipping_address['first_name'],
                    'last_name' => $cart->shipping_address['last_name'],
                    'email' => $payment->customer->email,
                    'company' => $cart->shipping_address['company'],
                    'street' => $cart->shipping_address['street'],
                    'street_number' => $cart->shipping_address['street_number'],
                    'zip' => $cart->shipping_address['zip'],
                    'city' => $cart->shipping_address['city'],
                    'country' => $cart->shipping_address['country'],
					'vat_number' => $cart->shipping_address['vat_number'] ?? null,
					'registration_number' => $cart->shipping_address['registration_number'] ?? null,
                    'billing_company' => $cart->billing_address['company'] ?? $cart->shipping_address['company'],
                    'billing_street' => $cart->billing_address['street'] ?? $cart->shipping_address['street'],
                    'billing_street_number' => $cart->billing_address['street_number'] ?? $cart->shipping_address['street_number'],
                    'billing_zip' => $cart->billing_address['zip'] ?? $cart->shipping_address['zip'],
                    'billing_city' => $cart->billing_address['city'] ?? $cart->shipping_address['city'],
                    'billing_country' => $cart->billing_address['country'] ?? $cart->shipping_address['country'],
					'billing_vat_number' => $cart->billing_address['vat_number'] ?? $cart->shipping_address['vat_number'] ?? null,
					'billing_registration_number' => $cart->billing_address['registration_number'] ?? $cart->shipping_address['registration_number'] ?? null,
                    'shipping_price' => $cart->getShippingMethodPrice(), // TODO: Save price in the cart in case it changed recently ?,
                    'payment_method_price' => 0,
                    'amount' => $payment->amount,
                    'currency' => $payment->currency,
                    'payment_method' => $cart->payment_method,
                    'shipping_method' => $cart->shipping_method,
                    'region' => $cart->region,
                    'status' => $orderStatus,
                ]);
            });

            // Create order items
            $cart->items->each(function ($item) use ($order) {
                $order->items()->create([
                    'reference' => $item->reference,
                    'name' => $item->name,
                    'quantity' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'total_tax_rate' => $item->taxes->sum('rate'),
                    'status' => OrderItemStatus::Pending,
                ]);

                if ($item->variant->track_stock) {
                    $item->variant->decrement('stock', $item->quantity);
                }
            });

            $cart->getTaxes()->each(fn ($tax) => $order->taxes()->create([
                'name' => $tax['name'],
                'tax_included' => $tax['tax_included'],
                'taxable_amount' => $tax['taxable_amount'],
                'rate' => $tax['rate'],
                'tax_amount' => $tax['amount'],
            ]));

            $cart->update([
                'order_id' => $order->id,
            ]);

            $payment->update([
                'order_id' => $order->id,
            ]);

			if ($payment->customer->locale !== app()->getLocale()) {
				$payment->customer->update([
					'locale' => app()->getLocale(),
				]);
			}

            return $order;
        });

        (new OrderObserver)->created($order);

        Log::info('Order created', [
            'order' => $order->toArray(),
        ]);

        return redirect()->route('shop::payment.confirmation', [
            'gateway' => $payment->gateway,
            'reference' => $payment->reference,
        ]);
    }
}
