<?php

namespace FirstpointCh\Shop\Observers;

use FirstpointCh\Shop\Events\OrderCancelled;
use FirstpointCh\Shop\Events\OrderCreated;
use FirstpointCh\Shop\Events\OrderPaid;
use FirstpointCh\Shop\Events\OrderShipped;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Order\Enums\OrderStatus;

class OrderObserver
{
    public function created(Order $order)
    {
        $order->activities()->create([
            'user_name' => 'System',
            'action_type' => 'flow',
            'action_title' => 'La commande a été créée',
        ]);

        OrderCreated::dispatch($order);

        match ($order->status) {
            OrderStatus::Paid => OrderPaid::dispatch($order),
            OrderStatus::Shipped => OrderShipped::dispatch($order),
            OrderStatus::Cancelled => OrderCancelled::dispatch($order),
            default => null,
        };
    }

    public function updated(Order $order)
    {
        if ($changes = $order->getChanges()) {
            if (array_key_exists('status', $changes)) {
                if (app()->runningInConsole()) {
                    $name = 'System';
                } elseif (request()->is('cp/*')) {
                    $name = auth(config('shop.cp.guard'))->user()?->name ?? 'System';
                } else {
                    $name = auth()->user()?->full_name ?? 'System';
                }

                if (OrderStatus::from($changes['status']) === OrderStatus::Paid) {
                    $this->onPaid($order, $name);
                } elseif (OrderStatus::from($changes['status']) === OrderStatus::Shipped) {
                    $this->onShipped($order, $name);
                } elseif (OrderStatus::from($changes['status']) === OrderStatus::Cancelled) {
                    $this->onCancelled($order, $name);
                }
            }
        }
    }

    private function onPaid(Order $order, string $name)
    {
        $order->activities()->create([
            'user_name' => $name,
            'action_type' => 'flow',
            'action_title' => 'La commande a été payée',
        ]);

        try { // In case there's an error in a listener
            OrderPaid::dispatch($order);
        } catch (\Exception $e) {
            //
        }
    }

    private function onShipped(Order $order, string $name)
    {
        $order->activities()->create([
            'user_name' => $name,
            'action_type' => 'flow',
            'action_title' => 'La commande a été expédiée',
        ]);

        try { // In case there's an error in a listener
            OrderShipped::dispatch($order);
        } catch (\Exception $e) {
            //
        }
    }

    private function onCancelled(Order $order, string $name)
    {
        $order->activities()->create([
            'user_name' => $name,
            'action_type' => 'flow',
            'action_title' => 'La commande a été annulée',
        ]);

        try { // In case there's an error in a listener
            OrderCancelled::dispatch($order);
        } catch (\Exception $e) {
            //
        }
    }
}
