<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Attribute\AsAttributeValues;
use FirstpointCh\Shop\Cart\Orderable;
use FirstpointCh\Shop\Catalog\QueryBuilders\VariantQueryBuilder;
use FirstpointCh\Shop\Models\CatalogIndex as Index;
use FirstpointCh\Shop\Traits\HasCustomFields;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\DB;

/**
 * @method static \FirstpointCh\Shop\Catalog\QueryBuilders\VariantQueryBuilder query()
 */
class Variant extends Model implements Orderable
{
    use HasPackageFactory, Translatable, SoftDeletes, HasCustomFields;

    protected $guarded = [];

    protected $casts = [
        'is_active' => 'boolean',
        'name' => Localized::class,
        'slug' => Localized::class,
        'track_stock' => 'boolean',
        'is_shippable' => 'boolean',
        'meta_title' => Localized::class,
        'meta_description' => Localized::class,
        'attribute_data' => AsAttributeValues::class,
    ];

    protected static function booted()
    {
        static::addGlobalScope('region-aware', function ($builder) {
            $builder->when(shop()->isRegionAware(), function ($query) {
                $query->whereRelation('prices', 'region', shop()->getRegion()->slug);
            });
        });
    }

    public function newEloquentBuilder($query)
    {
        return new VariantQueryBuilder($query);
    }

    public function resolveRouteBinding($value, $field = null)
    {
        if ($field === 'slug') {
            return $this->where('slug->'.app()->getLocale(), $value)->first()
                ?? $this->where('slug->'.config('app.fallback_locale'), $value)->firstOrFail();
        }

        return $this->resolveRouteBindingQuery($this, $value, $field)->first();
    }

    public function product(): BelongsTo
    {
        return $this->belongsTo(Product::class);
    }

    // public function attributes(): BelongsToMany
    // {
    //     return $this->belongsToMany(Attribute::class)
    //         ->withPivot(['type', 'value', 'translatable'])
    //         ->withTimestamps()
    //         ->using(AttributeVariant::class);
    // }

    public function prices(): HasMany
    {
        return $this->hasMany(VariantPrice::class);
    }

    public function loadPrice($region = null)
    {
        $price = $this->prices()
            ->where('region', $region ?? shop()->getRegion()->slug)
            ->first();

        if (! $price) {
            return null;
        }

        $this->setAttribute('price', $price->price);

        return $price;
    }

    public function isOrderable($quantity = 1): bool
    {
        return $this->is_active
            && ! $this->isOutOfStock($quantity)
            && $this->prices()->where('region', shop()->getRegion()->slug)->exists();
    }

    public function isOutOfStock($quantity = 1): bool
    {
        if (! $this->track_stock) {
            return false;
        }

        return $this->stock - $quantity < 0;
    }

    public function index(Product $product = null)
    {
        if ($this->trashed()) {
            return;
        }

        $product = $product ?? $this->product;

        DB::transaction(function () use ($product) {
            $defaultProduct = $product->in(config('app.fallback_locale'))->toArray();

            foreach ($this->prices as $price) {
                foreach (config('shop.locales') as $locale => $config) {
                    $product->in($locale);
                    $this->in($locale);

                    Index::withoutGlobalScope('region-aware')->updateOrCreate([
                        'id' => $price->region.'-'.$locale.'-'.$product->id.'-'.$this->id,
                    ], [
                        'product_id' => $product->id,
                        'region' => $price->region,
                        'locale' => $locale,
                        'variant_id' => $this->id,
                        'product_name' => $product->name ?? $defaultProduct['name'],
                        'product_status' => $product->status,
                        'product_categories' => $product->categories->pluck('path')->map(fn ($item) => explode('/', $item))->flatten()->unique()->values(),
                        'product_brand_id' => $product->brand_id,
                        'variant_sku' => $this->sku,
                        'variant_reference' => $this->reference,
                        'variant_name' => $this->name,
                        'variant_is_active' => $this->is_active,
                        'price' => $price->price,
                        'product_attributes' => $product->attribute_data,
                        'variant_attributes' => ! empty($this->attribute_data) ? $this->attribute_data->mapWithKeys(fn ($item, $key) => [
                            $key => $item->localized($locale),
                        ]) : [],
                    ]);
                }
            }
        });
    }
}
