<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Traits\HasPackageFactory;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class TaxRule extends Model
{
    use HasPackageFactory, Translatable;

    protected $guarded = [];

    protected $casts = [
        'name' => Localized::class,
        'description' => Localized::class,
        'rate' => 'decimal:2',
        'tax_included' => 'boolean',
        'apply_to_all_regions' => 'boolean',
        'apply_to_all_product_types' => 'boolean',
        'apply_to_shipping' => 'boolean',
    ];

    protected static function booted()
    {
        static::addGlobalScope('region-aware', function ($builder) {
            $builder->when(shop()->isRegionAware(), function ($query) {
                $query->where(function ($query) {
                    $query->where('apply_to_all_regions', true)->orWhereRelation('regions', 'region', shop()->getRegion()->slug);
                });
            });
        });
    }

    public function regions(): HasMany
    {
        return $this->hasMany(TaxRuleRegion::class);
    }

    public function productTypes(): BelongsToMany
    {
        return $this->belongsToMany(ProductType::class)
            ->withTimestamps();
    }

    public function products(): BelongsToMany
    {
        return $this->belongsToMany(Product::class)
            ->withTimestamps();
    }

    public function scopeForProduct($query, Product $product)
    {
        return $query
            ->where(function ($query) use ($product) {
                $query->where('apply_to_all_product_types', true)
                    ->orWhereHas('productTypes', function ($query) use ($product) {
                        $query->where('product_types.id', $product->product_type_id);
                    })
                    ->orWhereHas('products', function ($query) use ($product) {
                        $query->where('products.id', $product->id);
                    });
            });
    }

    // TODO: Different calculation if included ?
    public function calculate($price)
    {
        return $price * $this->rate / 100;
    }
}
