<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Catalog\QueryBuilders\ProductQueryBuilder;
use FirstpointCh\Shop\Traits\HasCustomFields;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;
use Spatie\MediaLibrary\MediaCollections\Models\Media;
use Staudenmeir\EloquentJsonRelations\HasJsonRelationships;
use Staudenmeir\EloquentJsonRelations\Relations\BelongsToJson;

/**
 * @method static \FirstpointCh\Shop\Catalog\QueryBuilders\ProductQueryBuilder query()
 */
class Product extends Model implements HasMedia
{
    use HasPackageFactory, Translatable, InteractsWithMedia, HasCustomFields, HasJsonRelationships;

    protected $guarded = [];

    protected $casts = [
        'is_active' => 'boolean',
        'name' => Localized::class,
        'slug' => Localized::class,
        'short_description' => Localized::class,
        'description' => Localized::class,
        'meta_title' => Localized::class,
        'meta_description' => Localized::class,
        'attribute_data' => 'collection',
        'pictures' => 'collection',
    ];

    protected static function booted()
    {
        static::addGlobalScope('region-aware', function ($builder) {
            $builder->when(shop()->isRegionAware(), function ($query) {
                $query->whereRelation('prices', 'region', shop()->getRegion()->slug);
            });
        });
    }

    public function newEloquentBuilder($query): ProductQueryBuilder
    {
        return new ProductQueryBuilder($query);
    }

    public function resolveRouteBinding($value, $field = null)
    {
        if ($field === 'slug') {
            return $this->where('slug->'.app()->getLocale(), $value)->first()
                ?? $this->where('slug->'.config('app.fallback_locale'), $value)->firstOrFail();
        }

        return $this->resolveRouteBindingQuery($this, $value, $field)->first();
    }

    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('pictures')
            ->useDisk('product_pictures')
            ->registerMediaConversions(function (Media $media) {
                $this->addMediaConversion('cp_thumbnail')->crop('crop-center', 150, 150);
                $this->addMediaConversion('cp_tiny')->crop('crop-center', 50, 50);

                foreach (config('shop.media.product_pictures.conversions') as $name => $conversion) {
                    $this->addMediaConversion($name)
                        ->crop('crop-center', $conversion['width'], $conversion['height'])
                        ->withResponsiveImages();
                }
            });
    }

    public function categories(): BelongsToMany
    {
        return $this->belongsToMany(Category::class);
    }

    public function brand(): BelongsTo
    {
        return $this->belongsTo(Brand::class);
    }

    public function taxRules(): BelongsToMany
    {
        return $this->belongsToMany(TaxRule::class)
            ->withTimestamps();
    }

    public function prices(): HasMany
    {
        return $this->hasMany(VariantPrice::class);
    }

    public function variants(): HasMany
    {
        return $this->hasMany(Variant::class)
            ->withPrice();
    }

    public function attributes()
    {
        return $this->belongsToJson(Attribute::class, 'attribute_data->*', 'key');
    }

    public function productType(): BelongsTo
    {
        return $this->belongsTo(ProductType::class);
    }

    public function assets(): BelongsToJson
    {
        return $this->belongsToJson(Asset::class, 'pictures->*.asset_id', 'id');
    }

    public function index()
    {
        \DB::transaction(function () {
            $this->variants()->withoutGlobalScope('region-aware')->get()->map->index($this);
        });
    }

    public function associatedProducts(): BelongsToMany
    {
        return $this->belongsToMany(Product::class, 'product_associations', 'product_id', 'associated_product_id')
            ->withPivot('type')
            ->withTimestamps();
    }

    public function associatedProducts2(): BelongsToMany
    {
        return $this->belongsToMany(Product::class, 'product_associations', 'associated_product_id', 'product_id')
            ->withPivot('type')
            ->withTimestamps();
    }
}
