<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Traits\HasCustomFields;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Category extends Model
{
    use HasPackageFactory, Translatable, HasCustomFields;

    protected $guarded = [];

    protected $casts = [
        'name' => Localized::class,
        'is_active' => 'boolean',
    ];

    protected $appends = [
        'slug',
    ];

    public function products(): BelongsToMany
    {
        return $this->belongsToMany(Product::class);
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(Category::class, 'parent_id');
    }

    public function children(): HasMany
    {
        return $this
            ->hasMany(Category::class, 'parent_id')
            ->where('is_active', true)
            ->with('children');
    }

    public function allChildren(): HasMany
    {
        return $this
            ->hasMany(Category::class, 'parent_id')
            ->with('allChildren');
    }

    public function flattenChildrenIds(): array
    {
        $this->loadMissing('children');

        $ids = [];

        foreach ($this->children as $child) {
            $ids[] = $child->id;
            $ids = array_merge($ids, $child->flattenChildrenIds());
        }

        return $ids;
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function getSlugAttribute()
    {
        return str(is_array($this->name) ? $this->name[config('app.fallback_locale')] : $this->name)->slug();
    }

    public function rebuildPath()
    {
        $category = $this;

        $path = [$category->slug];

        while ($category->parent()->exists()) {
            $category = $category->parent;
            $path[] = $category->slug;
        }

        $this->update([
            'path' => implode('/', array_reverse($path)),
        ]);
    }
}
