<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Traits\HasPackageFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class CartItem extends Model
{
    use HasPackageFactory;

    protected $guarded = [];

    protected $appends = ['is_orderable'];

    public $incrementing = false;

    protected $keyType = 'string';

    protected $casts = [
        'unit_price' => 'float',
        'meta' => 'json',
        'taxes' => 'collection',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->id)) {
                $model->id = Str::uuid()->toString();
            }
        });
    }

    public function cart(): BelongsTo
    {
        return $this->belongsTo(Cart::class);
    }

    public function variant()
    {
        return $this->belongsTo(Variant::class, 'reference', 'reference')->withPrice();
    }

    public function getIsOrderableAttribute(): bool
    {
        return $this->variant?->isOrderable($this->quantity) ?? false;
    }

    public function getPriceAttribute()
    {
        return $this->unit_price * $this->quantity;
    }

    public function getTaxesPriceAttribute()
    {
        return $this->taxes->sum('amount');
    }

    public function recalculateTaxes(): self
    {
        $this->update([
            'taxes' => TaxRule::query()
                ->forProduct($this->variant->product)
                ->get()
                ->map(fn (TaxRule $taxRule) => [
                    'tax_rule_id' => $taxRule->id,
                    'name' => $taxRule->name,
                    'tax_included' => $taxRule->tax_included,
                    'taxable_amount' => $this->unit_price * $this->quantity,
                    'rate' => $taxRule->rate,
                    'amount' => $taxRule->tax_included
                        ? $this->price * $taxRule->rate / (100 + $taxRule->rate)
                        : $this->price * $taxRule->rate / 100,
                ])
                ->values(),
        ]);

        return $this;
    }
}
