<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Shop\Exceptions\ProductIsNotAvailable;
use FirstpointCh\Shop\Payment\Gateway;
use FirstpointCh\Shop\Region;
use FirstpointCh\Shop\Traits\HasPackageFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;

class Cart extends Model
{
    use HasPackageFactory;

    protected $guarded = [];

    public $incrementing = false;

    protected $keyType = 'string';

    protected $casts = [
        'is_active' => 'boolean',
        'shipping_address' => 'array',
        'billing_address' => 'array',
        'taxes' => 'collection',
    ];

    public $totalTax;

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (empty($model->id)) {
                $model->id = Str::orderedUuid()->toString();
            }
        });
    }

    public function getRegionAttribute($value)
    {
        return Region::fromSlug($value);
    }

    public function items()
    {
        return $this->hasMany(CartItem::class)->whereHas('variant');
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function customer()
    {
        return $this->belongsTo(Customer::class);
    }

    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    public function add(int $id = null, string $reference = null, string $sku = null, int $quantity = 1)
    {
        if (! empty($id)) {
            $variant = Variant::withPrice()->find($id);
        } elseif (! empty($reference)) {
            $variant = Variant::withPrice()->where('reference', $reference)->first();
        } elseif (! empty($sku)) {
            $variant = Variant::withPrice()->where('sku', $sku)->first();
        } else {
            throw new \Exception('No id, reference or sku provided.');
        }

        if (is_null($variant) || ! $variant->is_active || $variant->product->status !== 'published') {
            throw new ProductIsNotAvailable('The product is not available.');
        }

        if (empty($variant->price)) {
            throw new ProductIsNotAvailable('This product is not available in this region.');
        }

        if (! $this->id) {
            $this->save();
            $this->refresh();

            session(['shop::cart' => $this->id]);
        }

        $item = $this->items()
            ->where('orderable_id', $variant->id)
            ->where('orderable_type', get_class($variant))
            ->first();

        if (! empty($item)) {
            $item->increment('quantity', $quantity);
        } else {
            $item = $this->items()->create([
                'name' => $variant->product->name,
                'orderable_id' => $variant->id,
                'orderable_type' => get_class($variant),
                'quantity' => $quantity,
                'meta' => null,
            ]);
        }

        return $item;
    }

    public function getSubtotal()
    {
        return $this->items->sum('price');
    }

    public function getTaxes(): Collection
    {
        return $this->items
            ->pluck('taxes')
            ->flatten(1)
            ->groupBy('tax_rule_id')
            ->map(fn ($taxes) => [
                'type' => 'product',
                'tax_rule_id' => $taxes->first()['tax_rule_id'],
                'name' => $taxes->first()['name'],
                'tax_included' => $taxes->first()['tax_included'],
                'taxable_amount' => $taxes->sum('taxable_amount'),
                'rate' => $taxes->first()['rate'],
                'amount' => $taxes->sum('amount'),
            ])->values()
            ->merge(collect($this->taxes))
            ->values();
    }

    public function getTotal()
    {
        $subtotal = $this->getSubTotal();
        $taxes = $this->getTaxes()->where('tax_included', false)->sum('amount');

        // TODO: Add VAT to shipping method price
        $shippingMethodPrice = $this->getShippingMethodPrice();
        $paymentMethodPrice = $this->getPaymentMethodPrice();

        return $subtotal + $taxes + $shippingMethodPrice + $paymentMethodPrice;
    }

    public function reset()
    {
        $this->update([
            'is_active' => false,
        ]);

        session()->forget('shop::cart');

        app('shop::cart')->reset();
    }

    public function getShippingMethodPrice()
    {
        if (empty($this->shipping_method)) {
            return 0;
        }

        return ShippingMethod::query()
            ->restrictByCartPrice($this->getSubtotal())
            ->withPrice($this->getSubtotal())
            ->where('slug', $this->shipping_method)
            ->first()->price;
    }

    // TODO: Dont use this method
    public function getPaymentMethodPrice()
    {
        if (empty($this->payment_method)) {
            return 0;
        }

        try {
            return app(Gateway::class)->make($this->payment_method)
                ->calculateFee($this);
        } catch (\Exception $e) {
            return 0;
        }
    }

    public function recalculateShippingTaxes()
    {
        if (! $this->shipping_method) {
            return $this;
        }

        $shippingMethod = ShippingMethod::query()
            ->restrictByCartPrice($this->getSubtotal())
            ->withPrice($this->getSubtotal())
            ->where('slug', $this->shipping_method)
            ->first();

        $taxRules = TaxRule::query()
            ->where('apply_to_shipping', true)
            ->get();

        if ($taxRules->isEmpty()) {
            return $this;
        }

        $taxes = ($this->taxes ?? collect())->filter(fn ($tax) => $tax['type'] !== 'shipping');

        foreach ($taxRules as $taxRule) {
            $taxes->push([
                'type' => 'shipping',
                'tax_rule_id' => $taxRule->id,
                'name' => $taxRule->name,
                'tax_included' => $taxRule->tax_included,
                'taxable_amount' => $shippingMethod->price,
                'rate' => $taxRule->rate,
                'amount' => $taxRule->tax_included
                    ? $shippingMethod->price * $taxRule->rate / (100 + $taxRule->rate)
                    : $shippingMethod->price * $taxRule->rate / 100,
            ]);
        }

        $this->update([
            'taxes' => $taxes,
        ]);

        return $this;
    }
}
