<?php

namespace FirstpointCh\Shop\Models;

use FirstpointCh\Translatable\Casts\Localized;
use FirstpointCh\Translatable\Traits\Translatable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Intervention\Image\ImageManager;

class Asset extends Model
{
    use Translatable;

    protected $guarded = [];

    protected $casts = [
        'display_name' => Localized::class,
        'alt_text' => Localized::class,
        'conversions' => 'collection',
    ];

    protected $appends = [
        'url',
        'conversion_urls',
        'filename',
    ];

    public function getUrlAttribute()
    {
        return Storage::disk($this->disk)->url($this->path);
    }

    public function getConversionUrlsAttribute()
    {
        return $this->conversions?->map(fn ($path) => Storage::disk(config('shop.media_conversion_disk'))->url($path)) ?? collect();
    }

    public function getFilenameAttribute()
    {
        return pathinfo($this->path, PATHINFO_FILENAME);
    }

    public function regenerateConversions(array $config = null, $overriteExisting = false)
    {
        if (! $this->mime_type || ! str_starts_with($this->mime_type, 'image/')) {
            return;
        }

        if (is_null($config)) {
            $config = config('shop.media.'.$this->disk.'.conversions');
        }

        if (empty($config)) {
            return;
        }

        $conversions = [];

        $config['cp_tiny'] = [
            'width' => 50,
            'height' => 50,
        ];

        $imageManager = (new ImageManager(['driver' => 'gd']))->make(Storage::disk($this->disk)->get($this->path));

        $imageManager->backup();

        $changed = false;

        foreach ($config as $name => $conversion) {
            $pathInfo = pathinfo($this->path);
            $pathInfo['dirname'] = $pathInfo['dirname'] === '.' ? '' : $pathInfo['dirname'].'/';

            $newPath = $pathInfo['dirname'].$pathInfo['filename'].'-'.$name.(! empty($pathInfo['extension']) ? '.'.$pathInfo['extension'] : '');

            if (! $overriteExisting && $this->conversions?->get($name) === $newPath) {
                continue;
            }

            $image = $imageManager->fit($conversion['width'], $conversion['height'], function ($constraint) {
                $constraint->upsize();
            });

            Storage::disk(config('shop.media_conversion_disk'))->put($newPath, $image->stream());

            $conversions[$name] = $newPath;

            $imageManager->reset();

            $changed = true;
        }

        if ($changed) {
            $this->update([
                'conversions' => $conversions,
            ]);
        }
    }

    public function move($newPath)
    {
        $ext = pathinfo($this->path, PATHINFO_EXTENSION);
        $newPath = $newPath . '/' . $this->filename . '.' . $ext;

        Storage::disk($this->disk)->move($this->path, $newPath);

        $this->update([
            'path' => $newPath,
        ]);
    }
}
