<?php

namespace FirstpointCh\Shop\Http\Requests;

use Illuminate\Contracts\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;

class DatatableRequest
{
    private $searchable = [];

    private $query;

    private $transformer = null;

    private $defaultSortColumn = 'id';

    private $defaultSortDirection = 'desc';

    private $orderBy = null;

    public function defaultSort(string $column, string $direction): self
    {
        $this->defaultSortColumn = $column;
        $this->defaultSortDirection = $direction;

        return $this;
    }

    public function searchable(array $searchable)
    {
        $this->searchable = $searchable;

        return $this;
    }

    public function query(Builder $query): self
    {
        $this->query = $query;

        return $this;
    }

    /**
     * @param  callable(Collection):self  $transformer
     * @return $this
     */
    public function transform(callable $transformer): self
    {
        $this->transformer = $transformer;

        return $this;
    }

    /**
     * @param  callable(Collection):self  $orderBy
     * @return $this
     */
    public function orderBy(callable $orderBy): self
    {
        $this->orderBy = $orderBy;

        return $this;
    }

    public function get()
    {
        if (empty($this->query)) {
            throw new \Exception('Base query not set');
        }

        if ($q = request('filters.search')) {
            $this->query->where(function (Builder $query) use ($q) {
                foreach ($this->searchable as $key => $value) {
                    if (is_callable($value)) {
                        $value($query, $q);
                    } else {
                        $temp = str_replace('\'', '\'\'', $q);

                        $query->orWhereRaw("LOWER({$value}) like LOWER('%{$temp}%')");
                    }
                }
            });
        }

        [$column, $direction] = request('sort.column')
            ? [request('sort.column'), request('sort.direction')]
            : [$this->defaultSortColumn, $this->defaultSortDirection];

        if (! empty($this->orderBy)) {
            ($this->orderBy)($this->query, $column, $direction);
        } else {
            $this->query->orderBy($column, $direction);
        }

        $results = $this->query->paginate(request('perPage', 10), ['*'], 'page');

        if ($this->transformer) {
            $results->setCollection(($this->transformer)($results->getCollection()));
        }

        return $results;
    }

    public function getState()
    {
        return collect([
            'filters' => request('filters'),
            'sort' => [
                'column' => request('sort.column', $this->defaultSortColumn),
                'direction' => request('sort.direction', $this->defaultSortDirection),
            ],
            'page' => (int) request('page'),
            'perPage' => (int) request('perPage'),
            'data' => $this->get(),
        ])->toArray();

        // $state = request()->only(['filters', 'sort', 'page', 'perPage']);

        // return ! empty($state) ? $state : null;
    }
}
