<?php

namespace FirstpointCh\Shop\Http\Controllers;

use FirstpointCh\Shop\Models\Payment;
use FirstpointCh\Shop\Order\Enums\OrderStatus;
use FirstpointCh\Shop\Payment\Actions\CreateOrderAction;
use FirstpointCh\Shop\Payment\Enums\PaymentStatus;
use FirstpointCh\Shop\Payment\PaymentGatewayManager;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class HandlePaymentController extends Controller
{
    public function __construct(
        public PaymentGatewayManager $gatewayManager
    ) {
    }

    public function __invoke($gateway, Request $request)
    {
        $paymentGateway = $this->gatewayManager->driver($gateway);

        $payment = $paymentGateway->handle($request);

        return match ($paymentGateway->handleMode) {
            'online' => $this->handleOnline($payment),
            'offline' => $this->handleOffline($payment),
            default => abort(403),
        };
    }

    private function handleOnline(Payment $payment)
    {
        Log::info('Payment status: '.$payment->id.' - '.$payment->status->getLabel());

        return match ($payment->status) {
            PaymentStatus::Completed => CreateOrderAction::run($payment, OrderStatus::Pending, $payment->cart),
            PaymentStatus::Authorized => CreateOrderAction::run($payment, OrderStatus::Paid, $payment->cart),
            PaymentStatus::Paid => CreateOrderAction::run($payment, OrderStatus::Paid, $payment->cart),
            PaymentStatus::Cancelled => redirect()->route(config('shop.cart_url'))->with('error', 'Payment cancelled'),
            PaymentStatus::Failed => redirect()->route(config('shop.cart_url'))->with('error', 'Payment failed'),
            PaymentStatus::Error => redirect()->route(config('shop.cart_url'))->with('error', 'Payment error'),
            PaymentStatus::Unknown => abort(403),
            default => abort(403),
        };
    }

    private function handleOffline(Payment $payment)
    {
        match ($payment->status) {
            PaymentStatus::Completed => CreateOrderAction::run($payment, OrderStatus::Pending, $payment->cart),
            PaymentStatus::Authorized => CreateOrderAction::run($payment, OrderStatus::Paid, $payment->cart),
            PaymentStatus::Paid => CreateOrderAction::run($payment, OrderStatus::Paid, $payment->cart),
            default => abort(403),
        };

        return response()->noContent();
    }
}
