<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use DB;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\ShippingMethod;
use FirstpointCh\Shop\Region;
use Illuminate\Support\Arr;
use Inertia\Inertia;

class ShippingMethodController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('ShippingMethod/Index', [
            'datatableState' => $datatable
                ->searchable(['name', 'description', 'slug'])
                ->query(ShippingMethod::query())
                ->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('ShippingMethod/Form', [
            'regions' => Region::all(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'prices' => 'nullable|array',
            'prices.*.region' => 'required',
            'prices.*.min_price' => 'nullable',
            'prices.*.max_price' => 'nullable',
            'prices.*.price' => 'required',
        ]);

        $prices = Arr::pull($data, 'prices');

        ShippingMethod::create($data)
            ->prices()
            ->createMany($prices);

        return redirect('/cp/shipping-methods');
    }

    public function edit(ShippingMethod $shippingMethod)
    {
        return Inertia::render('ShippingMethod/Form', [
            'shippingMethod' => $shippingMethod->in('*')->load('prices'),
            'regions' => Region::all(),
        ]);
    }

    public function update(ShippingMethod $shippingMethod)
    {
        $data = request()->validate([
            'name' => 'required|array',
            'description' => 'required|array',
            'slug' => 'required|max:255',
            'prices' => 'nullable|array',
            'prices.*.id' => 'nullable|exists:shipping_method_prices,id',
            'prices.*.region' => 'required',
            'prices.*.min_price' => 'nullable',
            'prices.*.max_price' => 'nullable',
            'prices.*.price' => 'required',
        ]);

        $prices = Arr::pull($data, 'prices');

        DB::transaction(function () use ($data, $prices, $shippingMethod) {
            $shippingMethod->update($data);

            $ids = Arr::whereNotNull(Arr::pluck($prices, 'id'));

            DB::table('shipping_method_prices')->whereNotIn('id', $ids)->delete();

            foreach ($prices as $price) {
                if (! empty($price['id'])) {
                    DB::table('shipping_method_prices')->where('id', $price['id'])->update([
                        'min_price' => $price['min_price'],
                        'max_price' => $price['max_price'],
                        'price' => $price['price'],
                    ]);
                } else {
                    $shippingMethod->prices()->create([
                        'region' => $price['region'],
                        'min_price' => $price['min_price'],
                        'max_price' => $price['max_price'],
                        'price' => $price['price'],
                    ]);
                }
            }
        });

        return redirect('/cp/shipping-methods');
    }
}
