<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use Arr;
use FirstpointCh\Shop\Data\Cp\DefaultVariantAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductVariantFormData;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\CustomField;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\TaxRule;
use FirstpointCh\Shop\Models\Variant;
use FirstpointCh\Shop\Region;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ProductVariantController extends Controller
{
    public function create(Product $product)
    {
        $variant = new Variant([
            'is_active' => false,
            'name' => [config('app.fallback_locale') => null],
            'sku' => null,
            'reference' => null,
            'slug' => [config('app.fallback_locale') => null],
            'is_shippable' => true,
            'weight_grams' => null,
            'track_stock' => false,
            'stock' => null,
            'attribute_data' => empty($product->productType)
                ? []
                : DefaultVariantAttributeData::collection($product->productType->variantAttributes),
            'prices' => [],
            'custom_fields' => [],
            'meta_title' => [config('app.fallback_locale') => null],
            'meta_description' => [config('app.fallback_locale') => null],
        ]);

        $variant->setRelation('customFields', CustomField::query()
            ->whereJsonContains('enabled_models', Variant::class)
            ->whereJsonContains('assigned_models', Variant::class)
            ->get()
            ->map(fn ($customField) => $customField->setAttribute('pivot', [
                'value' => $customField->translatable ? [config('app.fallback_locale') => null] : null,
            ])));

        $productAttributeKeys = $product->attribute_data;
        $productAttributes = Attribute::whereIn('key', $productAttributeKeys)->get();

        return Inertia::render('ProductVariant/Form', [
            'product' => $product,
            'variant' => $variant->in('*'),
            'attributes' => Attribute::with('options')->orderBy('name->'.config('app.fallback_locale'))->get(),
            'regions' => Region::all(),
            'taxRules' => TaxRule::all(),
            'customFields' => CustomField::whereJsonContains('enabled_models', Variant::class)->get(),
            'productAttributes' => $productAttributes,
        ]);
    }

    public function store(Product $product, ProductVariantFormData $data)
    {
        DB::transaction(function () use ($product, $data) {
            $variant = $product->variants()->create($data->getVariantData());

            foreach ($data->getPrices($product->id) as $price) {
                if (! is_null($price['price'])) {
                    $variant->prices()->create($price);
                }
            }

            $variant->updateCustomFields($data->custom_fields);
            $variant->index();
        });

        return redirect()->route('cp.products.show', [
            'product' => $product,
            'tab' => 'variants',
        ])->with('toast', [
            'title' => 'Variante créé',
            'message' => 'La variante a bien été créé',
            'type' => 'success',
        ]);
    }

    public function edit(Product $product, Variant $variant)
    {
        $variant->load(['prices', 'customFields']);
        $customFields = CustomField::query()
            ->whereJsonContains('enabled_models', Variant::class)
            ->whereJsonContains('assigned_models', Variant::class)
            ->get()
            ->map(function ($customField) use ($variant) {
                $foundIndex = $variant->customFields->search(fn ($item) => $item->id === $customField->id);
                $found = $foundIndex !== false ? $variant->customFields->pull($foundIndex) : null;

                return $found ?? $customField->setAttribute('pivot', [
                    'value' => $customField->translatable ? [config('app.fallback_locale') => null] : null,
                ]);
            });

        $variant->customFields->each(fn ($customField) => $customFields->push($customField));

        $variant->setRelation('customFields', $customFields);

        $productAttributeKeys = $product->attribute_data;
        $productAttributes = Attribute::whereIn('key', $productAttributeKeys)->get();

        return Inertia::render('ProductVariant/Form', [
            'product' => $product,
            'variant' => $variant->in('*'),
            'attributes' => Attribute::with('options')->orderBy('name->'.config('app.fallback_locale'))->get(),
            'regions' => Region::all(),
            'taxRules' => TaxRule::all(),
            'customFields' => CustomField::whereJsonContains('enabled_models', Variant::class)->get(),
            'productAttributes' => $productAttributes,
        ]);
    }

    public function update(Product $product, Variant $variant, ProductVariantFormData $data)
    {
        DB::transaction(function () use ($product, $variant, $data) {
            $variant->update($data->getVariantData());

            foreach ($data->getPrices($product->id) as $price) {
                if (! is_null($price['price'])) {
                    $region = Arr::pull($price, 'region');

                    $variant->prices()->updateOrCreate(['region' => $region], $price);
                } else {
                    $variant->prices()->where('region', $price['region'])->delete();
                }
            }

            $variant->prices()->whereIn('region', $data->getInactiveRegions())->delete();

            $variant->updateCustomFields($data->custom_fields);
            $variant->index();
        });

        return redirect()->route('cp.products.show', [
            'product' => $product,
            'tab' => 'variants',
        ])->with('toast', [
            'title' => 'Variante enregistrée',
            'message' => 'La variante a bien été enregistrée',
            'type' => 'success',
        ]);
    }

    public function delete(Product $product, Variant $variant)
    {
        $variant->delete();
        $variant->index();

        return redirect('/cp/products/'.$product->id.'/variants');
    }

    public function forceDelete(Product $product, $variant)
    {
        Variant::withTrashed()->find($variant)->forceDelete();

        return redirect('/cp/products/'.$product->id.'/variants');
    }

    public function restore(Product $product, $variant)
    {
        $variant = Variant::withTrashed()->find($variant);

        $variant->restore();
        $variant->index();

        return redirect('/cp/products/'.$product->id.'/variants');
    }

    public function replicate(Product $product, Variant $variant)
    {
        $variantCopy = DB::transaction(function () use ($product, $variant) {
            $variant->in('*');

            $variantCopy = $variant->replicate(['price'])->fill([
                'is_active' => false,
                'product_id' => $product->id,
                'name' => array_map(fn ($name) => $name.'-copie', $variant->name),
                'slug' => array_map(fn ($slug) => $slug.'-copie', $variant->slug),
                'reference' => $variant->reference.'-copie',
            ]);

            $variantCopy->save();

            $variant->prices->each(fn ($price) => $variantCopy->prices()->create([
                'product_id' => $product->id,
                'price' => $price->price,
                'region' => $price->region,
            ]));

            $variantCopy->updateCustomFields($variant->customFields->toArray());

            $product->variants()->save($variantCopy);

            return $variantCopy;
        });

        return redirect('/cp/products/'.$product->id.'/variants/'.$variantCopy->id.'/edit');
    }
}
