<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\ProductType;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ProductTypeController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('ProductType/Index', [
            'datatableState' => $datatable
                ->searchable(['name'])
                ->query(
                    ProductType::with([
                        'productAttributes' => fn ($query) => $query->select('name'),
                        'variantAttributes' => fn ($query) => $query->select('name'),
                    ])
                )
                ->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('ProductType/Form', [
            'availableAttributes' => Attribute::all()->sortByDesc('name')->values(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'name' => 'required',
            'productAttributes' => 'nullable|array',
            'variantAttributes' => 'nullable|array',
        ]);

        $productAttributes = collect(Arr::pull($data, 'productAttributes'))->mapWithKeys(function ($attributeId) {
            return [$attributeId => ['type' => 'product']];
        });

        $variantAttributes = collect(Arr::pull($data, 'variantAttributes'))->mapWithKeys(function ($attributeId) {
            return [$attributeId => ['type' => 'variant']];
        });

        DB::transaction(function () use ($data, $productAttributes, $variantAttributes) {
            $productType = ProductType::create($data);
            $productType->productAttributes()->sync($productAttributes->toArray());
            $productType->variantAttributes()->sync($variantAttributes->toArray());
        });

        return redirect('/cp/product-types');
    }

    public function edit(ProductType $productType)
    {
        $data = $productType->toArray();
        $data['productAttributes'] = $productType->productAttributes->pluck('id')->toArray();
        $data['variantAttributes'] = $productType->variantAttributes->pluck('id')->toArray();

        return Inertia::render('ProductType/Form', [
            'attr' => $data,
            'availableAttributes' => Attribute::all()->sortByDesc('name')->values(),
        ]);
    }

    public function update(ProductType $productType)
    {
        $data = request()->validate([
            'name' => 'required',
            'productAttributes' => 'nullable|array',
            'variantAttributes' => 'nullable|array',
        ]);

        $productAttributes = collect(Arr::pull($data, 'productAttributes'))->mapWithKeys(function ($attributeId) {
            return [$attributeId => ['type' => 'product']];
        });

        $variantAttributes = collect(Arr::pull($data, 'variantAttributes'))->mapWithKeys(function ($attributeId) {
            return [$attributeId => ['type' => 'variant']];
        });

        DB::transaction(function () use ($productType, $data, $productAttributes, $variantAttributes) {
            $productType->update($data);
            $productType->productAttributes()->sync($productAttributes->toArray());
            $productType->variantAttributes()->sync($variantAttributes->toArray());
        });

        return redirect('/cp/product-types');
    }
}
