<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Data\Cp\DefaultProductAttributeData;
use FirstpointCh\Shop\Data\Cp\ProductFormData;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Brand;
use FirstpointCh\Shop\Models\Category;
use FirstpointCh\Shop\Models\CustomField;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductAssociation;
use FirstpointCh\Shop\Models\ProductType;
use FirstpointCh\Shop\Models\TaxRule;
use FirstpointCh\Shop\Models\Variant;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class ProductController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Product/Index', [
            'datatableState' => $datatable
                ->searchable([
                    'id',
                    'name' => function ($query, $search) {
                        foreach (array_keys(config('shop.locales')) as $locale) {
                            $query->orWhereRaw('lower(json_unquote(json_extract(`name`, \'$."'.$locale.'"\'))) like lower(?)', ["%{$search}%"]);
                        }
                    },
                    'slug',
                    'reference' => fn ($query, $search) => $query
                        ->orWhereHas('variants', fn ($query) => $query
                            ->where('reference', 'like', "%{$search}%")
                        ),
                ])
                ->query(Product::with(['productType', 'assets']))
                ->transform(fn ($results) => $results
                        ->map(fn ($item) => $item
                            ->setAttribute('thumbnail', $item->assets->first()?->url)))
                ->getState(),
            'productTypes' => ProductType::all(),
        ]);
    }

    public function create()
    {
        $productType = ProductType::find(request('productTypeId'));

        $product = new Product([
            'status' => 'draft',
            'name' => [config('app.fallback_locale') => null],
            'slug' => [config('app.fallback_locale') => null],
            'product_type_id' => request('productTypeId'),
            'short_description' => [config('app.fallback_locale') => null],
            'description' => [config('app.fallback_locale') => null],
            'attribute_data' => empty($productType) ? [] : DefaultProductAttributeData::collection($productType->productAttributes),
            'brand_id' => null,
            'custom_fields' => [],
        ]);

        $product->setRelation('categories', collect());
        $product->setRelation('customFields', CustomField::query()
            ->whereJsonContains('enabled_models', Product::class)
            ->whereJsonContains('assigned_models', Product::class)
            ->get()
            ->map(fn ($customField) => $customField->setAttribute('pivot', [
                'value' => $customField->translatable ? [config('app.fallback_locale') => null] : null,
            ])));

        return Inertia::render('Product/Form', [
            'product' => $product->in('*'),
            'productTypes' => ProductType::all(),
            'categories' => Category::whereNull('parent_id')->with('allChildren')->get(),
            'brands' => Brand::orderBy('name')->get(),
            'customFields' => CustomField::whereJsonContains('enabled_models', Product::class)->get(),
            'attributes' => Attribute::with('options')->orderBy('name->'.config('app.fallback_locale'))->get(),
        ]);
    }

    public function store(ProductFormData $data)
    {
        $product = DB::transaction(function () use ($data) {
            $product = Product::create($data->getProductData());
            $product->categories()->sync($data->categories);
            $product->updateCustomFields($data->custom_fields);
            $product->index();

            return $product;
        });

        return redirect('/cp/products/'.$product->id)->with('toast', [
            'title' => 'Produit créé',
            'message' => 'Le produit a bien été créé',
            'type' => 'success',
        ]);
    }

    public function show(DatatableRequest $datatable, Product $product, string $tab = null)
    {
        $product->setRelation('categories', $product->categories->pluck('name'));

        if (is_null($product->pictures)) {
            $product->pictures = collect();
        }

        return Inertia::render('Product/Show', [
            'tab' => $tab,
            'product' => $product,
            'datatableState' => $datatable
                ->searchable(['name', 'slug'])
                ->query(
                    Variant::query()
                        ->with(['prices', 'customFields'])
                        ->where('product_id', $product->id)
                        ->when(request('filters.trashed') === 'yes', fn ($query) => $query->withTrashed())
                        ->when(request('filters.trashed') === 'only', fn ($query) => $query->onlyTrashed())
                )
                ->getState(),
            'taxRules' => (new DatatableRequest)->query(TaxRule::forProduct($product))->getState(),
            'associations' => (new DatatableRequest)->query(ProductAssociation::where('product_id', $product->id)->with('product', 'associatedProduct'))->getState(),
            'displayableCustomFields' => CustomField::whereJsonContains('enabled_models', Variant::class)->where('display_on_index', true)->get(),
        ]);
    }

    public function edit(Product $product)
    {
        $categories = Category::whereNull('parent_id')->with('allChildren')->get();

        $product->setRelation('categories', $product->categories->pluck('id'));
        $product->load(['customFields']);
        $customFields = CustomField::query()
            ->whereJsonContains('enabled_models', Product::class)
            ->whereJsonContains('assigned_models', Product::class)
            ->get()
            ->map(function ($customField) use ($product) {
                $foundIndex = $product->customFields->search(fn ($item) => $item->id === $customField->id);
                $found = $foundIndex !== false ? $product->customFields->pull($foundIndex) : null;

                return $found ?? $customField->setAttribute('pivot', [
                    'value' => $customField->translatable ? [config('app.fallback_locale') => null] : null,
                ]);
            });

        $product->customFields->each(fn ($customField) => $customFields->push($customField));

        $product->setRelation('customFields', $customFields);

        $data = $product->in('*')->toArray();

        if (empty($data['description'])) {
            $data['description'] = [
                config('app.fallback_locale') => null,
            ];
        }

        return Inertia::render('Product/Form', [
            'product' => $data,
            'productTypes' => ProductType::all(),
            'productTypeAttributes' => $product->productType?->productAttributes->map->only(['id', 'name', 'type']) ?? [],
            'attributes' => Attribute::with('options')->orderBy('name->'.config('app.fallback_locale'))->get(),
            'categories' => $categories,
            'brands' => Brand::orderBy('name')->get(),
            'customFields' => CustomField::whereJsonContains('enabled_models', Product::class)->get(),
        ]);
    }

    public function update(Product $product, ProductFormData $data)
    {
        DB::transaction(function () use ($product, $data) {
            $product->update($data->getProductData());
            $product->categories()->sync($data->getCategoryData());
            $product->updateCustomFields($data->custom_fields);
            $product->index();
        });

        return redirect('/cp/products/'.$product->id)->with('toast', [
            'title' => 'Produit enregistré',
            'message' => 'Le produit a bien été enregistré',
            'type' => 'success',
        ]);
    }

    public function updateStatus(Product $product)
    {
        $data = request()->validate([
            'status' => 'required|in:draft,published',
        ]);

        $product = DB::transaction(function () use ($product, $data) {
            $product->update($data);
            $product->index();

            return $product;
        });

        return back()->with('toast', [
            'title' => 'Produit mis à jour',
            'message' => 'Le produit a bien été '.['draft' => 'désactivé', 'published' => 'publié'][$product->status],
            'type' => 'success',
        ]);
    }

    public function delete(Product $product)
    {
        $product->delete();

        return redirect('/cp/products')->with('toast', [
            'title' => 'Produit supprimé',
            'message' => 'Le produit a bien été supprimé',
            'type' => 'success',
        ]);
    }

    public function replicate(Product $product)
    {
        $productData = $product->in(config('app.fallback_locale'))->toArray();
        $productData['status'] = 'draft';
        $productData['name'] .= '-copie';
        $productData['slug'] .= '-copie';

        unset($productData['id']);

        $productCopy = DB::transaction(function () use ($product, $productData) {
            $productCopy = Product::create($productData);

            $productCopy->categories()->sync($product->categories->pluck('id'));

            $productCopy->updateCustomFields(
                customFields: $product->customFields->toArray(),
                locales: [config('app.fallback_locale')],
            );

            $productCopy->variants()->saveMany($product->variants->map(function ($variant) use ($productCopy) {
                $variantData = $variant->in(config('app.fallback_locale'))->toArray();
                $variantData['product_id'] = $productCopy->id;
                $variantData['name'] .= '-copie';
                $variantData['slug'] .= '-copie';
                $variantData['reference'] .= '-copie';
                $variantData['attribute_data'] = $variant->attribute_data
                    ->map(
                        fn ($attr) => is_array($attr->value)
                            ? Arr::only($attr->value, [config('app.fallback_locale')])
                            : $attr->value
                    );

                unset($variantData['id']);
                unset($variantData['price']);

                $variantCopy = $variant->create($variantData);

                $variant->prices->each(fn ($price) => $variantCopy->prices()->create([
                    'product_id' => $productCopy->id,
                    'price' => $price->price,
                    'region' => $price->region,
                ]));

                $variantCopy->updateCustomFields($variant->customFields->toArray());

                return $variantCopy;
            }));

            return $productCopy;
        });

        return redirect('/cp/products/'.$productCopy->id)->with('toast', [
            'title' => 'Produit dupliqué',
            'message' => 'Le produit a bien été dupliqué',
            'type' => 'success',
        ]);
    }

    public function updatePictures(Product $product)
    {
        $data = request()->validate([
            'pictures' => 'nullable|array',
        ]);

        if (empty($data['pictures'])) {
            $data['pictures'] = null;
        }

        $product->update($data);

        return redirect('/cp/products/'.$product->id)->with('toast', [
            'title' => 'Produit mis à jour',
            'message' => 'Les images du produit ont bien été mises à jour',
            'type' => 'success',
        ]);
    }

    public function search()
    {
        return Product::where(function ($query) {
            foreach (array_keys(config('shop.locales')) as $locale) {
                $query->orWhereRaw('lower(json_unquote(json_extract(`name`, \'$."'.$locale.'"\'))) like lower(?)', ["%".request('search')."%"]);
            }
        })
        ->limit(10)
        ->get(['id', 'name']);
    }

    public function addAssociation(Product $product)
    {
        $data = request()->validate([
            'type' => 'required|in:related,similar',
            'product_id' => 'required|exists:products,id',
        ]);

        // Check if association already exists
        if ($product->associatedProducts()->where('associated_product_id', $data['product_id'])->exists()) {
            return back()->with('toast', [
                'title' => 'Association impossible',
                'message' => 'Cette association de produits existe déjà',
                'type' => 'error',
            ]);
        }

        $product->associatedProducts()->attach($data['product_id'], ['type' => $data['type']]);

        return back()->with('toast', [
            'title' => 'Association ajoutée',
            'message' => 'L\'association a bien été ajoutée',
            'type' => 'success',
        ]);
    }

    public function removeAssociation(Product $product, ProductAssociation $association)
    {
        $association->delete();

        return back()->with('toast', [
            'title' => 'Association supprimée',
            'message' => 'L\'association a bien été supprimée',
            'type' => 'success',
        ]);
    }
}
