<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use Carbon\CarbonImmutable;
use FirstpointCh\Shop\Actions\Asset\RefreshFromDisk;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Models\Asset;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Storage;

class MediaLibraryController extends Controller
{
    public function index()
    {
        $data = request()->validate([
            'disk' => 'nullable|string',
            'path' => 'nullable|string',
        ]);

        $data['disk'] ??= 'public';
        $data['path'] ??= null;

        if (Str::startsWith($data['path'], '/')) {
            $data['path'] = Str::after($data['path'], '/');
        }

        // TODO: Abort if disk is not in config('shop.media.*.media_library')

        $directories = collect(Storage::disk($data['disk'])->directories($data['path']))
            ->map(fn ($directory) => basename($directory))
            ->values();

        $files = Asset::query()
            ->where('disk', $data['disk'])
            ->when($data['path'], fn ($query) => $query->where('path', 'like', "{$data['path']}%"))
            ->get()
            ->filter(function ($asset) use ($data) {
                if (! Storage::disk($data['disk'])->exists($asset->path)) {
                    dd($asset->path);
                    $asset->delete();

                    return false;
                }

                return true;
            })->map(fn ($asset) => [
                'asset_id' => $asset->id,
                'display_name' => (object) $asset->raw('display_name'),
                'alt_text' => (object) $asset->raw('alt_text'),
                'directory' => Str::contains($asset->path, '/') ? Str::beforeLast($asset->path, '/') : '',
                'name' => basename($asset->path),
                'type' => $asset->mime_type,
                'url' => $asset->url,
                'last_modified' => CarbonImmutable::parse(Storage::disk($data['disk'])->lastModified($asset->path))->format('Y-m-d H:i:s'),
                'size' => number_format($asset->size / 1000, 2, '.', ''),
                'conversions' => $asset->conversion_urls,
            ])
            ->sortBy('asset_id')
            ->values();

        if (request()->wantsJson()) {
            return [
                'disk' => $data['disk'],
                'path' => $data['path'],
                'directories' => $directories,
                'files' => $files,
            ];
        } else {
            return inertia('MediaLibrary/Index', [
                'disk' => $data['disk'],
                'path' => $data['path'],
                'directories' => $directories,
                'files' => $files,
            ]);
        }
    }

    public function updateAsset()
    {
        $data = request()->validate([
            'disk' => 'required|string',
            'path' => 'nullable|string',
            'display_name' => 'nullable|array',
            'alt_text' => 'nullable|array',
        ]);

        Asset::updateOrCreate([
            'disk' => $data['disk'],
            'path' => $data['path'],
        ], $data);
    }

    public function getAssetsMetadata()
    {
        $data = request()->validate([
            'disk' => 'required|string',
            'assets' => 'required|array',
            'assets.*' => 'required|integer',
        ]);

        return Asset::where('disk', $data['disk'])
            ->find($data['assets'])
            ->map(fn ($asset) => [
                'asset_id' => $asset->id,
                'display_name' => (object) $asset->raw('display_name'),
                'alt_text' => (object) $asset->raw('alt_text'),
                'directory' => Str::contains($asset->path, '/') ? Str::beforeLast($asset->path, '/') : '',
                'path' => $asset->path,
                'name' => basename($asset->path),
                'type' => $asset->mime_type,
                'url' => $asset->url,
                'last_modified' => CarbonImmutable::parse(Storage::disk($data['disk'])->lastModified($asset->path))->format('Y-m-d H:i:s'),
                'size' => number_format($asset->size / 1000, 2, '.', ''),
            ])
            ->sortBy('asset_id')
            ->values();
    }

    public function deleteAsset()
    {
        $data = request()->validate([
            'disk' => 'required|string',
            'path' => 'required|string',
        ]);

        DB::transaction(function () use ($data) {
            Asset::where('disk', $data['disk'])
                ->where('path', $data['path'])
                ->delete();

            Storage::disk($data['disk'])->delete($data['path']);
        });
    }

    public function createDirectory()
    {
        $data = request()->validate([
            'disk' => 'required|string',
            'path' => 'nullable|string',
            'name' => 'required|string',
        ]);

        Storage::disk($data['disk'])->makeDirectory($data['path'].'/'.$data['name']);
    }

    public function refreshAssets(RefreshFromDisk $action)
    {
        $data = request()->validate([
            'disk' => 'required|string',
        ]);

        $action->run($data['disk']);
    }

    public function regenerateConversions()
    {
        $data = request()->validate([
            'asset_id' => 'required|exists:assets,id',
        ]);

        $asset = Asset::findOrFail($data['asset_id']);

        $asset->regenerateConversions(null, true);
    }

    public function moveFile()
    {
        $data = request()->validate([
            'disk' => 'required|string',
            'files.*' => 'required|exists:assets,id',
            'path' => 'nullable|string',
        ]);

        $path = $data['path'] ?? '';

        Asset::whereIn('id', $data['files'])->each(function ($asset) use ($path) {
            $asset->move($path);
        });
    }

    public function deleteFiles()
    {
        $data = request()->validate([
            'disk' => 'required|string',
            'files.*' => 'required|exists:assets,id',
        ]);

        Asset::whereIn('id', $data['files'])->each(function ($asset) {
            Storage::disk($asset->disk)->delete($asset->path);
            $asset->delete();
        });
    }

    public function renameFile(Asset $asset)
    {
        $data = request()->validate([
            'new_name' => 'required|string',
        ]);

        $newPath = Str::contains($asset->path, '/')
            ? Str::beforeLast($asset->path, '/').'/'.$data['new_name']
            : $data['new_name'];

        Storage::disk($asset->disk)->move($asset->path, $newPath);

        $asset->update([
            'path' => $newPath,
        ]);
    }
}
