<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use Arr;
use DB;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Category;
use FirstpointCh\Shop\Models\CustomField;
use Inertia\Inertia;

class CategoryController extends Controller
{
    public function tree()
    {
        return Inertia::render('Category/IndexTree', [
            'categories' => Category::whereNull('parent_id')->with('allChildren')->get(),
        ]);
    }

    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Category/Index', [
            'datatableState' => $datatable
                ->defaultSort('name', 'asc')
                ->orderBy(function ($query, $column, $direction) {
                    if ($column === 'name') {
                        $query->orderByRaw('lower(json_unquote(json_extract(`'.$column.'`, \'$."'.config('app.fallback_locale').'"\'))) '.$direction);
                    } else {
                        $query->orderBy($column, $direction);
                    }
                })
                ->searchable(['name'])
                ->query(Category::query())
                ->getState(),
        ]);
    }

    public function create()
    {
        $categories = Category::whereNull('parent_id')->with('allChildren')->get();

        return Inertia::render('Category/Form', [
            'categories' => $categories,
            'customFields' => CustomField::whereJsonContains('enabled_models', Category::class)->get(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'key' => 'required',
            'name' => 'required',
            'is_active' => 'required',
            'parent_id' => 'nullable',
            'custom_fields' => 'nullable|array',
        ]);

        $customFields = Arr::pull($data, 'custom_fields', []);

        DB::transaction(function () use ($data, $customFields) {
            $category = Category::create($data);
            $category->updateCustomFields($customFields);
            $category->rebuildPath();
        });

        return redirect('/cp/categories');
    }

    public function edit(Category $category)
    {
        $category->load('customFields');

        $customFields = CustomField::query()
            ->whereJsonContains('enabled_models', Category::class)
            ->whereJsonContains('assigned_models', Category::class)
            ->get()
            ->map(function ($customField) use ($category) {
                $foundIndex = $category->customFields->search(fn ($item) => $item->id === $customField->id);
                $found = $foundIndex !== false ? $category->customFields->pull($foundIndex) : null;

                return $found ?? $customField->setAttribute('pivot', [
                    'value' => $customField->translatable ? [config('app.fallback_locale') => null] : null,
                ]);
            });

        $category->customFields->each(fn ($customField) => $customFields->push($customField));

        $category->setRelation('customFields', $customFields);

        $categories = Category::whereNull('parent_id')->with('allChildren')->get();

        return Inertia::render('Category/Form', [
            'category' => $category->in('*'),
            'categories' => $categories,
            'customFields' => CustomField::whereJsonContains('enabled_models', Category::class)->get(),
        ]);
    }

    public function update(Category $category)
    {
        $data = request()->validate([
            'id' => 'required',
            'key' => 'required',
            'name' => 'required',
            'is_active' => 'required',
            'parent_id' => 'nullable|exists:categories,id|different:id',
            'custom_fields' => 'nullable|array',
        ]);

        DB::transaction(function () use ($category, $data) {
            $category->updateCustomFields(Arr::pull($data, 'custom_fields', []));
            $category->update($data);
            $category->rebuildPath();
        });

        return redirect('/cp/categories');
    }

    public function delete(Category $category)
    {
        $category->delete();

        return redirect('/cp/categories');
    }
}
