<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Brand;
use FirstpointCh\Shop\Models\CustomField;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class BrandController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Brand/Index', [
            'datatableState' => $datatable
                ->defaultSort('name', 'asc')
                ->searchable(['name'])
                ->query(Brand::query())
                ->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('Brand/Form', [
            'customFields' => CustomField::whereJsonContains('enabled_models', Brand::class)->get(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'name' => 'required',
            'custom_fields' => 'nullable|array',
        ]);

        $customFields = Arr::pull($data, 'custom_fields', []);

        DB::transaction(function () use ($data, $customFields) {
            $brand = Brand::create($data);
            $brand->updateCustomFields($customFields);
        });

        return redirect('/cp/brands');
    }

    public function edit(Brand $brand)
    {
        $brand->load('customFields');

        $customFields = CustomField::query()
            ->whereJsonContains('enabled_models', Brand::class)
            ->whereJsonContains('assigned_models', Brand::class)
            ->get()
            ->map(function ($customField) use ($brand) {
                $foundIndex = $brand->customFields->search(fn ($item) => $item->id === $customField->id);
                $found = $foundIndex !== false ? $brand->customFields->pull($foundIndex) : null;

                return $found ?? $customField->setAttribute('pivot', [
                    'value' => $customField->translatable ? [config('app.fallback_locale') => null] : null,
                ]);
            });

        $brand->customFields->each(fn ($customField) => $customFields->push($customField));

        $brand->setRelation('customFields', $customFields);

        return Inertia::render('Brand/Form', [
            'brand' => $brand,
            'customFields' => CustomField::whereJsonContains('enabled_models', Brand::class)->get(),
        ]);
    }

    public function update(Brand $brand)
    {
        $data = request()->validate([
            'name' => 'required',
            'custom_fields' => 'nullable|array',
        ]);

        DB::transaction(function () use ($brand, $data) {
            $brand->updateCustomFields(Arr::pull($data, 'custom_fields', []));
            $brand->update($data);
        });

        return redirect('/cp/brands');
    }

    public function delete(Brand $brand)
    {
        $brand->delete();

        return redirect('/cp/brands');
    }
}
