<?php

namespace FirstpointCh\Shop\Http\Controllers\Cp;

use DB;
use FirstpointCh\Shop\Http\Controllers\Controller;
use FirstpointCh\Shop\Http\Requests\DatatableRequest;
use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\CustomField;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Validation\Rule;
use Inertia\Inertia;

class AttributeController extends Controller
{
    public function index(DatatableRequest $datatable)
    {
        return Inertia::render('Attribute/Index', [
            'datatableState' => $datatable
                ->defaultSort('name', 'asc')
                ->orderBy(function ($query, $column, $direction) {
                    if ($column === 'name') {
                        $query->orderByRaw('lower(json_unquote(json_extract(`'.$column.'`, \'$."'.config('app.fallback_locale').'"\'))) '.$direction);
                    } else {
                        $query->orderBy($column, $direction);
                    }
                })
                ->searchable(['name'])
                ->query(Attribute::query())
                ->getState(),
        ]);
    }

    public function create()
    {
        return Inertia::render('Attribute/Form', [
            'customFieldTypes' => app('shop::shop')->getAttributeTypes(),
            'customFields' => CustomField::all(),
        ]);
    }

    public function store()
    {
        $data = request()->validate([
            'key' => 'required|unique:attributes,key|max:255',
            'name' => 'required',
            'type' => 'required',
            'translatable' => 'boolean',
            'filterable' => 'boolean',
            'options' => 'nullable|array',
            'custom_fields' => 'nullable|array',
        ]);

        if (in_array($data['type'], ['options', 'number'])) {
            $data['translatable'] = false;
        }

        DB::transaction(function () use ($data) {
            $options = Arr::pull($data, 'options', []);
            $customFields = Arr::pull($data, 'custom_fields');

            $attribute = Attribute::create($data);
            $attribute->options()->createMany($options);

            if (! empty($customFields)) {
                $attribute->updateCustomFields($customFields);
            }
        });

        return redirect('/cp/attributes');
    }

    public function edit(Attribute $attribute)
    {
        $attribute->load(['customFields', 'options']);
        $attribute->options->map->in('*');

        return Inertia::render('Attribute/Form', [
            'attr' => $attribute->in('*'),
            'customFieldTypes' => app('shop::shop')->getAttributeTypes(),
            'customFields' => CustomField::all(),
        ]);
    }

    public function update(Attribute $attribute)
    {
        $data = request()->validate([
            'key' => [
                'required',
                Rule::unique('attributes', 'key')->ignoreModel($attribute),
                'max:255',
            ],
            'name' => 'required',
            'type' => 'required',
            'translatable' => 'boolean',
            'filterable' => 'boolean',
            'options' => 'array',
            'custom_fields' => 'nullable|array',
        ]);

        DB::transaction(function () use ($attribute, $data) {
            $options = Arr::pull($data, 'options');
            $customFields = Arr::pull($data, 'custom_fields');

            $attribute->update($data);
            $attribute->updateCustomFields($customFields);

            foreach ($options as $option) {
                if (! empty($option['id'])) {
                    $attribute->options()->find($option['id'])->update($option);
                } else {
                    $attribute->options()->create($option);
                }
            }
        });

        Cache::forget('attributes');

        return redirect('/cp/attributes');
    }
}
