<?php

namespace FirstpointCh\Shop\Data\Cp;

use Illuminate\Validation\Rule;
use Spatie\LaravelData\Data;

class ProductVariantFormData extends Data
{
    public function __construct(
        public ?bool $is_active,
        public array $name,
        public array $slug,
        public string $reference,
        public bool $track_stock,
        public ?string $sku,
        public ?int $stock,
        public bool $is_shippable = true,
        public ?float $weight_grams = null,
        public ?array $attribute_data = [],
        public ?array $prices = [],
        public ?array $custom_fields = [],
        public ?array $meta_title = [],
        public ?array $meta_description = [],
    ) {
    }

    public static function rules(): array
    {
        $fallbackLocale = config('app.fallback_locale');

        return [
            'is_active' => 'nullable',
            'name.'.$fallbackLocale => 'required',
            'slug.'.$fallbackLocale => [
                'required',
                Rule::unique('variants', 'slug->'.$fallbackLocale)->ignore(request()->route('variant')?->id),
            ],
            'reference' => 'required',
            'track_stock' => 'required|boolean',
            'sku' => 'nullable',
            'stock' => 'nullable|numeric',
            'is_shippable' => 'boolean',
            'weight_grams' => 'nullable|numeric',
            'attribute_data' => 'nullable|array',
            'attribute_data.*.id' => 'required|exists:attributes,id',
            'attribute_data.*.value' => 'nullable',
            'prices' => 'array',
            'prices.*.region' => 'required',
            'prices.*.is_active' => 'nullable|boolean',
            'prices.*.price' => 'required_if:prices.*.is_active,1',
            'custom_fields' => 'nullable|array',
            'meta_title' => 'nullable|array',
            'meta_description' => 'nullable|array',
        ];
    }

    public function getVariantData()
    {
        return [
            'is_active' => $this->is_active,
            'name' => array_filter($this->name, fn ($value) => ! empty($value)),
            'slug' => array_filter($this->slug, fn ($value) => ! empty($value)),
            'reference' => $this->reference,
            'track_stock' => $this->track_stock,
            'sku' => $this->sku,
            'stock' => $this->stock,
            'is_shippable' => $this->is_shippable,
            'weight_grams' => $this->weight_grams,
            'meta_title' => array_filter($this->meta_title, fn ($value) => ! empty($value)),
            'meta_description' => array_filter($this->meta_description, fn ($value) => ! empty($value)),
            'attribute_data' => $this->getAttributes(),
        ];
    }

    public function getAttributes()
    {
        return collect($this->attribute_data)
            ->mapWithKeys(fn ($attr) => [$attr['key'] => $attr['value']])
            ->toArray();
    }

    public function getInactiveRegions()
    {
        return collect($this->prices)
            ->where('is_active', false)
            ->pluck('region')
            ->toArray();
    }

    public function getPrices($productId)
    {
        return collect($this->prices)
            ->where('is_active', true)
            ->map(fn ($price) => [
                'region' => $price['region'],
                'product_id' => $productId,
                'price' => $price['price'],
            ]);
    }
}
