<?php

namespace FirstpointCh\Shop\Catalog\QueryBuilders;

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\VariantPrice;
use Illuminate\Database\Eloquent\Builder;

class ProductQueryBuilder extends Builder
{
    public function withMinPrice(): self
    {
        return $this->addSelect(['min_price' => VariantPrice::selectRaw('MIN(price)')
            ->where('region', shop()->getRegion()->slug)
            ->whereColumn('variant_prices.product_id', 'products.id')
            ->join('variants', 'variants.id', '=', 'variant_prices.variant_id')
            ->where('variants.is_active', true)
            ->whereNull('variants.deleted_at')
            ->limit(1),
        ]);
    }

    public function published(): self
    {
        return $this->where('status', 'published');
    }

    public function whereCategoriesIn(?array $categories = []): self
    {
        if (empty($categories)) {
            return $this;
        }

        return $this->whereHas('categories', fn ($query) => $query->whereIn('categories.id', $categories));
    }

    public function wherePriceBetween(float $min = null, float $max = null): self
    {
        return $this->whereHas('prices', function ($query) use ($min, $max) {
            $query
                ->when(shop()->isRegionAware(), fn ($query) => $query->where('region', shop()->getRegion()->slug))
                ->when($min, fn ($query) => $query->where('price', '>=', $min))
                ->when($max, fn ($query) => $query->where('price', '<=', $max));
        });
    }

    public function whereAttributeIn(string $attribute, array $values, string $locale = null): self
    {
        $locale = $locale ?? app()->getLocale();

        return $this->whereHas('attributes', function ($query) use ($attribute, $values, $locale) {
            $query->where('attributes.key', $attribute);

            $query->where(function ($query) use ($values, $locale) {
                foreach ($values as $value) {
                    $query->orWhere('attribute_product.value->'.$locale, $value);
                }
            });
        });
    }

    public function whereAttribute(string|array $attribute, $value = null): self
    {
        if (is_array($attribute)) {
            foreach ($attribute as $key => $value) {
                $this->whereAttribute($key, $value);
            }

            return $this;
        }

        $attribute = Attribute::where('filterable', true)->where('key', $attribute)->first();

        if (empty($attribute)) {
            return $this;
        }

        return $this->whereHas('attributes', function ($query) use ($attribute, $value) {
            $query->where('attributes.key', $attribute->key);

            if (is_array($value)) {
                if ($attribute->translatable) {
                    $query->where(fn ($q) => $q
                        ->whereIn('attribute_product.value->'.app()->getLocale(), $value)
                        ->orWhereIn('attribute_product.value->'.config('app.fallback_locale'), $value));
                } else {
                    $query->whereIn('attribute_product.value', $value);
                }
            } else {
                if ($attribute->translatable) {
                    $query->where(fn ($q) => $q
                        ->where('attribute_product.value->'.app()->getLocale(), $value)
                        ->orWhere('attribute_product.value->'.config('app.fallback_locale'), $value));
                } else {
                    $query->where('attribute_product.value', $value);
                }
            }
        });
    }

    public function withTotalTaxRate($type = null): self
    {
        return $this->addSelect(['total_tax_rate' => function ($query) use ($type) {
            $query->selectRaw('SUM(tax_rules.rate)')
                ->from('tax_rules')
                ->when($type === 'included', function ($query) {
                    $query->where('tax_rules.tax_included', true);
                })
                ->when($type === 'excluded', function ($query) {
                    $query->where('tax_rules.tax_included', false);
                })
                ->where(function ($query) {
                    // The tax rule must be applied to all regions or the current region
                    $query->where('tax_rules.apply_to_all_regions', true)
                        ->orWhereExists(function ($query) {
                            $query->selectRaw(1)
                                ->from('tax_rule_regions')
                                ->whereColumn('tax_rule_regions.tax_rule_id', 'tax_rules.id')
                                ->where('tax_rule_regions.region', shop()->getRegion()->slug);
                        });
                })
                ->where(function ($query) {
                    // The tax rule must be applied to all product types or this product type
                    $query->where('tax_rules.apply_to_all_product_types', true)
                        ->orWhereExists(function ($query) {
                            $query->selectRaw(1)
                                ->from('product_type_tax_rule')
                                ->whereColumn('product_type_tax_rule.tax_rule_id', 'tax_rules.id')
                                ->whereColumn('product_type_tax_rule.product_type_id', 'products.product_type_id');
                        });
                });
        }])->withCasts([
            'total_tax_rate' => 'float',
        ]);
    }
}
