<?php

namespace FirstpointCh\Shop\Cart\Drivers;

use FirstpointCh\Shop\Cart\CartDriver;
use FirstpointCh\Shop\Cart\SessionCartItem;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Region;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;

class SessionCart implements CartDriver
{
    private string $identifier;

    public string $region;

    public ?Collection $content = null;

    public array $shipping_address = [];

    public array $billing_address = [];

    public ?string $shipping_method = null;

    public ?string $payment_method = null;

    public ?Order $order = null;

    public function init(string $region)
    {
        $this->region = $region;

        $data = session('shop::cart::'.$region);

        $this->identifier = $data->identifier ?? (string) Str::uuid();
        $this->content = collect($data->content ?? []);
        $this->shipping_address = $data->shipping_address ?? [];
        $this->billing_address = $data->billing_address ?? [];
        $this->shipping_method = $data->shipping_method ?? null;
        $this->payment_method = $data->payment_method ?? null;

        $this->save();
    }

    public function instance(): CartDriver
    {
        return $this;
    }

    public function getCartIdentifier()
    {
        return $this->identifier;
    }

    public function getRegion(): Region
    {
        return Region::fromSlug($this->region);
    }

    public function get(string $itemId)
    {
        return $this->content->get($itemId);
    }

    public function content(): Collection
    {
        return $this->content;
    }

    public function add(string $reference, string $name, float $unitPrice, ?int $quantity = 1, ?array $meta = [])
    {
        $item = $this->content->firstWhere('reference', $reference);

        if ($item) {
            $this->increment($item->id, $quantity);
        } else {
            $id = (string) Str::uuid();

            $item = new SessionCartItem(
                $id,
                $reference,
                $name,
                $unitPrice,
                $quantity,
                $meta,
            );

            $item->recalculateTaxes();

            $this->content->put($id, $item);
        }

        $this->save();
    }

    public function update(string $itemId, array $data)
    {
        $item = $this->content->get($itemId);

        $item->name = $data['name'] ?? $item->name;
        $item->unit_price = $data['unitPrice'] ?? $item->unit_price;
        $item->quantity = $data['quantity'] ?? $item->quantity;
        $item->meta = $data['meta'] ?? $item->meta;

        $item->recalculateTaxes();

        $this->content->put($itemId, $item);

        $this->save();
    }

    public function increment(string $itemId, ?int $quantity = 1)
    {
        $item = $this->content->get($itemId);

        $item->quantity += $quantity;

        $item->recalculateTaxes();

        $this->content->put($itemId, $item);

        $this->save();
    }

    public function decrement(string $itemId, ?int $quantity = 1)
    {
        $item = $this->content->get($itemId);

        $item->quantity -= $quantity;

        if ($item->quantity <= 0) {
            $this->remove($itemId);

            return;
        }

        $item->recalculateTaxes();

        $this->content->put($itemId, $item);

        $this->save();
    }

    public function remove(string $itemId)
    {
        $this->content->forget($itemId);

        $this->save();
    }

    public function save()
    {
        session()->put('shop::cart::'.$this->region, $this);
    }

    public function destroy()
    {
        session()->forget('shop::cart::'.$this->region);
    }

    public function getSubtotal(): float
    {
        return $this->content->sum(fn ($item) => $item->price);
    }

    public function getTaxes(): Collection
    {
        return $this->content
            ->pluck('taxes')
            ->flatten(1)
            ->groupBy('tax_rule_id')
            ->map(fn ($taxes) => [
                'tax_rule_id' => $taxes->first()['tax_rule_id'],
                'name' => $taxes->first()['name'],
                'tax_included' => $taxes->first()['tax_included'],
                'taxable_amount' => $taxes->sum('taxable_amount'),
                'rate' => $taxes->first()['rate'],
                'amount' => $taxes->sum('amount'),
            ])->values();
    }

    public function getTotal(): float
    {
        return $this->getSubtotal() + $this->getTaxes()->where('tax_included', false)->sum('amount');
    }

    public function shippingAddressIsValid(): bool
    {
        return isset($this->shipping_address['first_name'])
            && isset($this->shipping_address['last_name'])
            && isset($this->shipping_address['street'])
            && isset($this->shipping_address['street_number'])
            && isset($this->shipping_address['zip'])
            && isset($this->shipping_address['city'])
            && isset($this->shipping_address['country']);
    }

    public function shippingMethodIsValid(): bool
    {
        // TODO: Check that the shipping method still exists
        return isset($this->shipping_method);
    }

    public function paymentMethodIsValid(): bool
    {
        // TODO: Check that the payment method still exists
        return isset($this->payment_method);
    }

    public function getShippingAddress(): array
    {
        return $this->shipping_address;
    }

    public function setShippingAddress($firstName, $lastName, $company, $street, $streetNumber, $zip, $city, $country, $vatNumber = null, $registrationNumber = null, $hasBillingAddress = false)
    {
        $this->shipping_address = [
            'first_name' => $firstName,
            'last_name' => $lastName,
            'company' => $company,
            'street' => $street,
            'street_number' => $streetNumber,
            'zip' => $zip,
            'city' => $city,
            'country' => $country,
            'vat_number' => $vatNumber,
            'registration_number' => $registrationNumber,
            'has_billing_address' => $hasBillingAddress,
        ];

        $this->save();
    }

    public function getBillingAddress(): array
    {
        return $this->billing_address;
    }

    public function setBillingAddress($firstName, $lastName, $street, $streetNumber, $zip, $city, $country, $vatNumber = null, $registrationNumber = null, $company = null)
    {
        $this->billing_address = [
            'first_name' => $firstName,
            'last_name' => $lastName,
            'company' => $company,
            'street' => $street,
            'street_number' => $streetNumber,
            'zip' => $zip,
            'city' => $city,
            'country' => $country,
			'vat_number' => $vatNumber,
			'registration_number' => $registrationNumber
        ];

        $this->save();
    }

    public function getShippingMethod(): ?string
    {
        return $this->shipping_method;
    }

    public function setShippingMethod(string $shippingMethod)
    {
        $this->shipping_method = $shippingMethod;

        $this->save();
    }

    public function getPaymentMethod(): ?string
    {
        return $this->payment_method;
    }

    public function setPaymentMethod(string $paymentMethod)
    {
        $this->payment_method = $paymentMethod;

        $this->save();
    }

    public function setOrder(Order $order)
    {
        $this->order = $order;

        $this->save();
    }
}
