<?php

namespace FirstpointCh\Shop\Cart\Drivers;

use FirstpointCh\Shop\Cart\Cart as CartFacade;
use FirstpointCh\Shop\Cart\CartDriver;
use FirstpointCh\Shop\Models\Cart;
use FirstpointCh\Shop\Models\Customer;
use FirstpointCh\Shop\Models\Order;
use FirstpointCh\Shop\Models\Variant;
use FirstpointCh\Shop\Region;
use Illuminate\Support\Collection;

class DatabaseCart implements CartDriver
{
    public string $region;

    public ?Cart $cart;

    public Customer $user;

    public function init(string $region, Customer $user)
    {
        $this->region = $region;
        $this->user = $user;
        $this->cart = $user->cart;

		if ($this->cart?->order_id) {
            $this->newCart();
		}

        if (CartFacade::driver('session')->content()->isNotEmpty()) {
            CartFacade::driver('session')->content()->each(function ($item) {
                if (Variant::where('reference', $item->reference)->exists()) {
                    $this->add(
                        $item->reference,
                        $item->name,
                        $item->unit_price,
                        $item->quantity,
                        $item->meta ?? null,
                    );
                }
            });

            CartFacade::driver('session')->destroy();
        }
    }

    public function instance(): CartDriver
    {
        return $this;
    }

    public function newCart()
    {
        $this->user
            ->carts()
            ->where('region', $this->region)
            ->update([
                'is_active' => false,
            ]);

        $this->cart = $this->user->carts()->create([
            'customer_id' => auth()->id(),
            'region' => $this->region,
            'is_active' => true,
            'shipping_address' => [],
            'billing_address' => [],
        ]);
    }

    public function getCartModel()
    {
        return $this->cart;
    }

    public function getCartIdentifier()
    {
        return $this->cart->id;
    }

    public function __get($key)
    {
        if (isset($this->cart->{$key})) {
            return $this->cart->{$key};
        }

        return $this->{$key};
    }

    public function getRegion(): Region
    {
        return Region::fromSlug($this->region);
    }

    public function get(string $itemId)
    {
        return $this->cart?->items()->find($itemId);
    }

    public function content(): Collection
    {
        return $this->cart?->items ?? collect();
    }

    public function add(string $reference, string $name, float $unitPrice, ?int $quantity = 1, ?array $meta = [])
    {
        if (! $this->cart) {
            $this->newCart();
        }

        if ($item = $this->cart->items()->where('reference', $reference)->first()) {
            $item->increment('quantity', $quantity);
            $item->recalculateTaxes();
        } else {
            $this->cart->items()->create([
                'reference' => $reference,
                'name' => $name,
                'unit_price' => $unitPrice,
                'quantity' => $quantity,
                'meta' => $meta,
            ])->recalculateTaxes();
        }
    }

    public function update(string $itemId, array $data)
    {
        if (! $this->cart) {
            $this->newCart();
        }

        if ($item = $this->get($itemId)) {
            $item->update($data);
            $item->recalculateTaxes();
        }
    }

    public function increment(string $itemId, ?int $quantity = 1)
    {
        if (! $this->cart) {
            $this->newCart();
        }

        if ($item = $this->get($itemId)) {
            $item->increment('quantity', $quantity);
            $item->recalculateTaxes();
        }
    }

    public function decrement(string $itemId, ?int $quantity = 1)
    {
        if (! $this->cart) {
            $this->newCart();
        }

        if ($item = $this->get($itemId)) {
            $item->decrement('quantity', $quantity);
            $item->recalculateTaxes();
        }
    }

    public function remove(string $itemId)
    {
        if (! $this->cart) {
            $this->newCart();
        }

        $this->get($itemId)?->delete();
    }

    public function destroy()
    {
        if ($this->cart) {
            $this->cart->update([
                'is_active' => false,
            ]);
        }
    }

    public function getSubtotal(): float
    {
        return $this->content()->sum(fn ($item) => $item->price);
    }

    public function getTaxes(): Collection
    {
        return $this->cart?->getTaxes() ?? collect();
    }

    public function getTotal(): float
    {
        return $this->cart?->getTotal() ?? 0;
    }

    public function shippingAddressIsValid(): bool
    {
        if (! $this->cart) {
            return false;
        }

        return isset($this->cart->shipping_address['first_name'])
            && isset($this->cart->shipping_address['last_name'])
            && isset($this->cart->shipping_address['street'])
            && isset($this->cart->shipping_address['street_number'])
            && isset($this->cart->shipping_address['zip'])
            && isset($this->cart->shipping_address['city'])
            && isset($this->cart->shipping_address['country']);
    }

    public function billingAddressIsValid(): bool
    {
        if (! $this->cart) {
            return false;
        }

        if (empty($this->cart->shipping_address['has_billing_address'])) {
            return true;
        }

        return isset($this->cart->billing_address['first_name'])
            && isset($this->cart->billing_address['last_name'])
            && isset($this->cart->billing_address['street'])
            && isset($this->cart->billing_address['street_number'])
            && isset($this->cart->billing_address['zip'])
            && isset($this->cart->billing_address['city'])
            && isset($this->cart->billing_address['country']);
    }

    public function shippingMethodIsValid(): bool
    {
        // TODO: Check that the shipping method still exists
        return $this->cart && isset($this->cart->shipping_method);
    }

    public function paymentMethodIsValid(): bool
    {
        // TODO: Check that the payment method still exists
        return $this->cart && isset($this->cart->payment_method);
    }

    public function getShippingAddress(): array
    {
        return $this->cart?->shipping_address ?? [];
    }

    public function setShippingAddress($firstName, $lastName, $company, $street, $streetNumber, $zip, $city, $country, $vatNumber = null, $registrationNumber = null, $hasBillingAddress = false)
    {
        $this->cart->update([
            'shipping_address' => [
                'first_name' => $firstName,
                'last_name' => $lastName,
                'company' => $company,
                'street' => $street,
                'street_number' => $streetNumber,
                'zip' => $zip,
                'city' => $city,
                'country' => $country,
				'vat_number' => $vatNumber,
				'registration_number' => $registrationNumber,
                'has_billing_address' => $hasBillingAddress,
            ],
        ]);
    }

    public function getBillingAddress(): array
    {
        return $this->cart?->billing_address ?? [];
    }

    public function setBillingAddress($firstName, $lastName, $street, $streetNumber, $zip, $city, $country, $vatNumber = null, $registrationNumber = null, $company = null)
    {
        $this->cart->update([
            'billing_address' => [
                'first_name' => $firstName,
                'last_name' => $lastName,
                'company' => $company,
                'street' => $street,
                'street_number' => $streetNumber,
                'zip' => $zip,
                'city' => $city,
                'country' => $country,
				'vat_number' => $vatNumber,
				'registration_number' => $registrationNumber,
            ],
        ]);
    }

    public function getShippingMethod(): ?string
    {
        return $this->cart?->shipping_method;
    }

    public function setShippingMethod(string $shippingMethod)
    {
        $this->cart->update([
            'shipping_method' => $shippingMethod,
        ]);

        $this->cart->recalculateShippingTaxes();
    }

    public function getPaymentMethod(): ?string
    {
        return $this->cart?->payment_method;
    }

    public function setPaymentMethod(string $paymentMethod)
    {
        $this->cart->update([
            'payment_method' => $paymentMethod,
        ]);
    }

    public function setOrder(Order $order)
    {
        $this->cart->update([
            'order_id' => $order->id,
        ]);
    }
}
