<?php

namespace FirstpointCh\Shop\Actions;

use FirstpointCh\Shop\Models\Payment;
use LaravelDaily\Invoices\Classes\Buyer;
use LaravelDaily\Invoices\Classes\InvoiceItem;
use LaravelDaily\Invoices\Classes\Party;
use LaravelDaily\Invoices\Facades\Invoice;
use LaravelDaily\Invoices\Invoice as InvoicesInvoice;

class GenerateInvoiceFromPayment
{
    use Action;

    public function handle(Payment $payment, string $template = 'default-invoice'): InvoicesInvoice
    {
        $numberPattern = config('shop.invoice_number_pattern');

        $numberPattern = str($numberPattern)
            ->replace('{Y}', $payment->created_at->format('Y'))
            ->replace('{M}', $payment->created_at->format('m'))
            ->replace('{D}', $payment->created_at->format('d'));

        $padding = substr_count($numberPattern, '#');

        $numberPattern = str_replace('{'.str_repeat('#', $padding).'}', '{SEQUENCE}', $numberPattern);

        $invoice = Invoice::make($payment->order->reference)
            ->status($payment->status->getLabel())
            ->series('')
            ->sequencePadding($padding)
            ->sequence($payment->id)
            ->serialNumberFormat($numberPattern)
            ->template($template)
            ->currencyCode($payment->currency)
            ->currencySymbol($payment->currency);

        $invoice->buyer(new Buyer([
            'name' => $payment->order->first_name.' '.$payment->order->last_name,
            'address' => $payment->order->street.' '.$payment->order->street_number.', '.$payment->order->zip.' '.$payment->order->city,
            'payment_method' => $payment->gateway,
			'custom_fields' => [
                'first_name' => $payment->order->first_name,
                'last_name' => $payment->order->last_name,
                'company' => $payment->order->company,
                'email' => $payment->order->email,
                'street' => $payment->order->street,
                'street_number' => $payment->order->street_number,
                'zip' => $payment->order->zip,
                'city' => $payment->order->city,
                'country' => $payment->order->country,
                'registration_number' => $payment->order->registration_number,
                'vat_number' => $payment->order->vat_number,
                'id' => $payment->order->customer_id,
				'address_billing' => [
					'company' => $payment->order->billing_company,
					'street' => $payment->order->billing_street,
					'street_number' => $payment->order->billing_street_number,
					'zip' => $payment->order->billing_zip,
					'city' => $payment->order->billing_city,
					'country' => $payment->order->billing_country
				]
            ],
        ]));

        $invoice->seller(new Party([
            'name' => config('shop.company.name'),
            'address' => config('shop.company.street').' '.config('shop.company.street_number').', '.config('shop.company.zip').' '.config('shop.company.city'),
            'custom_fields' => [
                'email' => config('shop.company.email'),
                'street' => config('shop.company.street'),
                'street_number' => config('shop.company.street_number'),
                'zip' => config('shop.company.zip'),
                'city' => config('shop.company.city'),
                'country' => config('shop.company.country'),
            ],
        ]));

        foreach ($payment->order->items as $item) {
            $invoiceItem = (new InvoiceItem())
                ->title($item->name)
                ->description($item->reference)
                ->pricePerUnit($item->unit_price)
                ->quantity($item->quantity);

            if (! empty($item->total_tax_rate)) {
                $invoiceItem->taxByPercent($item->total_tax_rate);
            }

            $invoice->addItem($invoiceItem);
        }

        $invoice
            // ->discountByPercent(10)
            // ->taxRate(15)
            ->shipping($payment->order->shipping_price ?? 0)
			->date($payment->created_at);

        return $invoice;
    }
}
