<?php

use FirstpointCh\Shop\Models\Attribute;
use FirstpointCh\Shop\Models\Product;
use FirstpointCh\Shop\Models\ProductType;
use FirstpointCh\Shop\Models\Region;
use FirstpointCh\Shop\Models\Variant;
use Inertia\Testing\AssertableInertia;

it('shows the variant edit form', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->get('/cp/products/'.$product->id.'/variants/'.$variant->id.'/edit')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ProductVariant/Form')
            ->where('product.id', $product->id)
            ->where('variant.id', $variant->id)
        );
});

it('has default attributes based on product type', function () {
    $productType = ProductType::factory()->create();
    $attribute = Attribute::factory()->create();

    Attribute::factory()->create(); // This attribute should not be added by default

    $productType->variantAttributes()->attach($attribute, ['type' => 'variant']);

    $product = Product::factory()->create([
        'product_type_id' => $productType->id,
    ]);

    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->get('/cp/products/'.$product->id.'/variants/'.$variant->id.'/edit')
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ProductVariant/Form')
            ->has('variant.base_attributes', 1)
            ->whereAll([
                'variant.base_attributes.0.attribute_id' => $attribute->id,
                'variant.base_attributes.0.name' => $attribute->name,
            ])
        );
});

// TODO: Test that the default attribute value is propertly set according to the attribute type

it('has required fields', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'base_attributes' => [[]],
        'extra_attributes' => [[]],
        'regions' => [[]],
    ])->assertSessionHasErrors([
        'name',
        'sku',
        'slug',
        'track_stock',
        'base_attributes.0.attribute_id',
        'extra_attributes.0.attribute_id',
        'regions.0.id',
    ]);
});

test('track_stock must be boolean', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, ['track_stock' => 'test'])
        ->assertSessionHasErrors(['track_stock']);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, ['track_stock' => true])
        ->assertSessionDoesntHaveErrors(['track_stock']);
});

test('each attribute must exist', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);
    $attribute = Attribute::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'base_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
        'extra_attributes' => [
            ['attribute_id' => $attribute->id],
            ['attribute_id' => 5],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['base_attributes.0.attribute_id', 'extra_attributes.0.attribute_id'])
    ->assertSessionHasErrors(['base_attributes.1.attribute_id', 'extra_attributes.1.attribute_id']);
});

test('each attribute value must exist', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);
    $attribute = Attribute::factory()->create();
    $value = $attribute->options()->create(['value' => 'test']);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'base_attributes' => [
            ['value_option_id' => $value->id],
            ['value_option_id' => 14],
        ],
        'extra_attributes' => [
            ['value_option_id' => $value->id],
            ['value_option_id' => 14],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['base_attributes.0.value_option_id', 'extra_attributes.0.value_option_id'])
    ->assertSessionHasErrors(['base_attributes.1.value_option_id', 'extra_attributes.1.value_option_id']);
});

test('each region must exist', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);
    $region = Region::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'regions' => [
            ['id' => $region->id],
            ['id' => 5],
        ],
    ])
    ->assertSessionDoesntHaveErrors(['regions.0.id'])
    ->assertSessionHasErrors(['regions.1.id']);
});

test('price is required for every active regions', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);
    $region = Region::factory()->create();

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'regions' => [
            ['id' => $region->id, 'is_active' => 1, 'price' => null],
            ['id' => $region->id, 'is_active' => 0, 'price' => null],
        ],
    ])
    ->assertSessionHasErrors(['regions.0.price'])
    ->assertSessionDoesntHaveErrors(['regions.1.price']);
});

it('updates a product variant', function () {
    $product = Product::factory()->create();
    $variant = Variant::factory()->create([
        'product_id' => $product->id,
    ]);
    $baseAttribute = Attribute::factory()->create();
    $baseAttributeValue = $baseAttribute->options()->create(['value' => 'test']);
    $extraAttribute = Attribute::factory()->create();
    $extraAttributeValue = $extraAttribute->options()->create(['value' => 'test extra']);

    $this->post('/cp/products/'.$product->id.'/variants/'.$variant->id, [
        'is_active' => false,
        'name' => ['en' => 'Test'],
        'sku' => 'test',
        'slug' => ['en' => 'test'],
        'track_stock' => false,
        'stock' => null,
        'base_attributes' => [
            ['attribute_id' => $baseAttribute->id, 'value_option_id' => $baseAttributeValue->id],
        ],
        'extra_attributes' => [
            ['attribute_id' => $extraAttribute->id, 'value_option_id' => $extraAttributeValue->id],
        ],
        'regions' => [
            ['id' => defaultRegion()->id, 'is_active' => 1, 'price' => 129],
        ],
    ])
    ->assertStatus(302)
    ->assertRedirect('/cp/products/'.$product->id.'/variants');

    $this->assertDatabaseHas('variants', [
        'product_id' => $product->id,
        'is_active' => false,
        'name' => json_encode(['en' => 'Test']),
        'sku' => 'test',
        'track_stock' => false,
        'stock' => null,
        'slug' => json_encode(['en' => 'test']),
    ]);

    $variant = Variant::first();

    $variant->load('attributes');

    $this->assertCount(2, $variant->getRelation('attributes'));
    $this->assertEquals($baseAttributeValue->id, $variant->getRelation('attributes')->first()->pivot->value_option_id);
    $this->assertEquals($extraAttributeValue->id, $variant->getRelation('attributes')->last()->pivot->value_option_id);
    $this->assertCount(1, $variant->regions);
    $this->assertEquals(129, $variant->regions->first()->pivot->price);
});
