<?php

use FirstpointCh\Shop\Models\ShippingMethod;
use Illuminate\Support\Arr;
use Inertia\Testing\AssertableInertia;

it('shows the shipping method edit form', function () {
    $shippingMethod = ShippingMethod::factory()->create();

    $this->get('/cp/shipping-methods/'.$shippingMethod->id.'/edit')
        ->assertStatus(200)
        ->assertInertia(fn (AssertableInertia $page) => $page
            ->component('ShippingMethod/Form')
            ->where('shippingMethod.id', $shippingMethod->id)
            ->has('regions', 2)
        );
});

test('the name is required and must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods/'.$id, ['name' => 'test'])->assertStatus(302)->assertSessionHasErrors('name');
    $this->post('/cp/shipping-methods/'.$id, ['name' => ['en' => 'test']])->assertSessionDoesntHaveErrors('name');
});

test('the description is required and must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods/'.$id, ['description' => 'test'])->assertStatus(302)->assertSessionHasErrors('description');
    $this->post('/cp/shipping-methods/'.$id, ['description' => ['en' => 'test']])->assertSessionDoesntHaveErrors('description');
});

test('the slug is required', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id)->assertStatus(302)->assertSessionHasErrors('slug');
});

test('prices must be an array', function () {
    $id = ShippingMethod::factory()->create()->id;

    $this->post('/cp/shipping-methods/'.$id, ['prices' => 'test'])->assertStatus(302)->assertSessionHasErrors('prices');
    $this->post('/cp/shipping-methods/'.$id, ['prices' => []])->assertStatus(302)->assertSessionDoesntHaveErrors('prices');
});

it('updates a shipping method', function () {
    $shippingMethod = ShippingMethod::factory()->create();
    $shippingMethod->prices()->create([
        'region' => 'ch',
        'min_price' => 11,
        'max_price' => 22,
        'price' => 33,
    ]);

    $data = [
        'name' => ['en' => 'My shipping method'],
        'description' => ['en' => 'My shipping method'],
        'slug' => 'my-shipping-method',
        'prices' => [
            ['region' => 'ch', 'min_price' => null, 'max_price' => 50, 'price' => 123],
            ['region' => 'ch', 'min_price' => 51, 'max_price' => 100, 'price' => 200],
        ],
    ];

    $this->assertDatabaseCount('shipping_method_prices', 1);

    $this->post('/cp/shipping-methods/'.$shippingMethod->id, $data)
        ->assertStatus(302)
        ->assertRedirect('/cp/shipping-methods');

    $data['name'] = json_encode($data['name']);
    $data['description'] = json_encode($data['description']);
    $data['id'] = $shippingMethod->id;

    unset($data['prices']);

    $this->assertDatabaseCount('shipping_method_prices', 2);
    $this->assertDatabaseHas('shipping_methods', $data);
    $this->assertDatabaseHas('shipping_method_prices', [
        'region' => 'ch',
        'shipping_method_id' => $shippingMethod->id,
        'min_price' => null,
        'max_price' => 50,
        'price' => 123,
    ]);
    $this->assertDatabaseHas('shipping_method_prices', [
        'region' => 'ch',
        'shipping_method_id' => $shippingMethod->id,
        'min_price' => 51,
        'max_price' => 100,
        'price' => 200,
    ]);
    $this->assertDatabaseMissing('shipping_method_prices', [
        'region' => 'ch',
        'shipping_method_id' => $shippingMethod->id,
        'min_price' => 11,
        'max_price' => 22,
        'price' => 33,
    ]);
});
